'use client'

import { create } from 'zustand'
import { persist } from 'zustand/middleware'

interface User {
  id: string
  email?: string | null
  name?: string | null
  role?: string
  isApproved?: number
  company?: string
  tiktokAccessToken?: string | null
  youtubeAccessToken?: string | null
  instagramAccessToken?: string | null
  image?: string | null
  onboardingComplete?: number
}

interface AuthState {
  user: User | null
  accessToken: string | null
  refreshToken: string | null
  isLoading: boolean
  setAuth: (user: User, accessToken: string, refreshToken: string) => void
  updateUser: (updates: Partial<User>) => void
  logout: () => void
  refreshAccessToken: () => Promise<string | null>
}

export const useAuth = create<AuthState>()(
  persist(
    (set, get) => ({
      user: null,
      accessToken: null,
      refreshToken: null,
      isLoading: true,
      setAuth: (user, accessToken, refreshToken) => set({ user, accessToken, refreshToken }),
      logout: () => set({ user: null, accessToken: null, refreshToken: null }),
      updateUser: (updates: Partial<User>) => {
        const { user } = get()
        if (user) {
          set({ user: { ...user, ...updates } })
        }
      },
      refreshAccessToken: async () => {
        const { refreshToken } = get()
        if (!refreshToken) return null

        try {
          const response = await fetch('/api/auth/refresh', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ refreshToken })
          })

          if (!response.ok) {
            get().logout() // Token invalide, déconnexion
            return null
          }

          const data = await response.json()
          set({ accessToken: data.accessToken, refreshToken: data.refreshToken })
          return data.accessToken
        } catch (error) {
          console.error('Failed to refresh token:', error)
          get().logout()
          return null
        }
      },
    }),
    {
      name: 'auth-storage',
      onRehydrateStorage: () => (state) => {
        state?.setAuth(state.user!, state.accessToken!, state.refreshToken!) // Trigger re-render if needed
        if (state) {
          (state as AuthState).isLoading = false
        }
      }
    }
  )
)
