import { google } from 'googleapis'
import { Readable } from 'stream'

const SCOPES = ['https://www.googleapis.com/auth/drive']

// Configuration Google Drive
const GOOGLE_CLIENT_EMAIL = process.env.GOOGLE_CLIENT_EMAIL || 'swiplay-drive@lustrous-gamma-478321-h4.iam.gserviceaccount.com'
const GOOGLE_PRIVATE_KEY = process.env.GOOGLE_PRIVATE_KEY || `-----BEGIN PRIVATE KEY-----\nMIIEvQIBADANBgkqhkiG9w0BAQEFAASCBKcwggSjAgEAAoIBAQCxO6Q9q3eeHjWl\n7SdpnrIqOMO9flBgleHkCem8DrJC5csc1Et27ytulbwlaKjaKo9ZgIJlKU5YXguz\nN1v6M/hLqGWv0rnGNGef4IwZCXRRUcgQ5K6bPO1VWvuTpGXk4STXOx6hQ3fd8AXL\nRo+WW3GRGJqlXw9ZwkHd7EZ1lG4E4/9waePgjyMxlzSjTUK2hklqK1xo3BxcRghj\ngVdYuXOm6LYllOhLjU+HJ9bDpYyRs5erfekH2lYzraH1VL0vCYgV34NekdvE0/g6\neW5L3BwklYFZiVzATtDjWuIAqJXeHpfs+pKz9oVUsULr4h1STQAVNaSifpSAb69z\n8mGxf4khAgMBAAECggEAG/wYYMUjM/F+NBnLGDIph2GDhXdyoUWZWk9hu5gwd+qz\n6cnwXofA+mBPSd2Jko/s0aDpNf2nNBcpeXNmJQ7cB7TqgDA3CupkawoNj7h+nYaF\nyiZpwZ5bCpgSqEs+7Dx2pjuHXcb69YLRpLlwBmRK/YepMi/dNdA8DprLvK4iucez\nGcQYBbuPSUWqff13Sv31WXY9xCcfQRjMHdNMjfh20JMpbFVlIRv8r4KK9n8oZcW4\nLCYvnb3fhS/VGpyVdSJ1VS6U8necIQkezb+8cYWwNOu6pkDT1r2emABiex4OhzZ\n+I6Xj9VGq/XI0KQOKBeLAYHLxM8e6z7FJupJyV0zRwKBgQD3qZOgjlx2d6odt63h\nOJTNln6q4ElArL1+wZ1fHi4zRE5H3G6gyp/XGZIBuGhHX73ky0uu7e1XrHlk3GJE\n17CxTz6AYOPquKoMM/PU+IhfhxvLi/Vz8NtwscFL5UNJZhh5LOKf5gFUxMuEXtr5\nPqEEcznINNNhSDPHq9QT/daPhwKBgQC3MxWh2QaeaB/u8FcoUVqV9gv2+kNWhByP\ndI8YHULZzzPX+k240J7Srt+N3IrGa1zpT7JPGldSeTqgqIpEx2h7bWt07spISHdl\nNRKIAHKTtjlTaykqDsmOnJUwapFI03saaCiJc9wAYK9Lwtmgc3eogFTuEp3cO8iQ\nV6ZJfQM8FwKBgA9zxCM7XqeBldqe0DmCJz6giiytK8TeQkB7Yg9COc23EE9JgJmF\nqu1nygVTdSYm4hk5WfKDI9DZkQVq/rKe5oZZlGO8J9xIQ0vZnfgGvbOAIYrUo67l\nMzvXy2Cp4EXfGd3NoCIr3p9kW/JZXL46Wx+E9PD54spqD9kAbo4Ew3NXAoGAOq1r\nDVbU7uHM8JwcXnaIPCX9uRJ5hPl1qGEeAAxHQDpEkbXDNSHk7PhD9E0iSeFTdgsn\nwnxeYbUgmRrChrsT9Qyb8W0Wb0Z+r3TrRZJXIIaqH9Tf72+jEUGH62UGubu63z9j\nVHs+iQRuwSZczC6PsfHAwixGUab4W3sncLxQz+sCgYEAgMlsMRY0MQDqUj2HjXcD\nMI7zdzzI3oX2DrO4eUhgw6IWfuTvin79oOgskjDzO/qPt/zJcP+JLPbjYPvUwWTg\nemgTNXQ/dxtDXskUwfLJ+fUTB4JmCHord6J+ZGRHyhxGhCozCm1ZCNcVo6kIyAyW\nhVyfdI2yvr6cQfCBAFDc+WY=\n-----END PRIVATE KEY-----\n`
const GOOGLE_DRIVE_FOLDER_ID = process.env.GOOGLE_DRIVE_FOLDER_ID || '1CFDaQmfysCZ-0UBUz1809mnDZx2OHbf5'

if (!GOOGLE_CLIENT_EMAIL || !GOOGLE_PRIVATE_KEY || !GOOGLE_DRIVE_FOLDER_ID) {
  console.warn('Google Drive configuration incomplete. Uploads will be disabled.')
}

class GoogleDriveService {
  private drive: any = null

  constructor() {
    if (GOOGLE_CLIENT_EMAIL && GOOGLE_PRIVATE_KEY) {
      const auth = new google.auth.GoogleAuth({
        credentials: {
          client_email: GOOGLE_CLIENT_EMAIL,
          private_key: GOOGLE_PRIVATE_KEY.replace(/\\n/g, '\n')
        },
        scopes: SCOPES
      })

      this.drive = google.drive({ version: 'v3', auth })
    }
  }

  async uploadFile(
    fileBuffer: Buffer,
    fileName: string,
    mimeType: string,
    folderId?: string
  ): Promise<{ success: boolean; fileId?: string; url?: string; error?: string }> {
    if (!this.drive) {
      return {
        success: false,
        error: 'Google Drive not configured'
      }
    }

    try {
      const fileMetadata = {
        name: fileName,
        parents: folderId ? [folderId] : [GOOGLE_DRIVE_FOLDER_ID!]
      }

      const media = {
        mimeType,
        body: Readable.from(fileBuffer)
      }

      const response = await this.drive.files.create({
        resource: fileMetadata,
        media: media,
        fields: 'id,webViewLink'
      })

      // Rendre le fichier accessible publiquement
      await this.drive.permissions.create({
        fileId: response.data.id,
        requestBody: {
          role: 'reader',
          type: 'anyone'
        }
      })

      return {
        success: true,
        fileId: response.data.id,
        url: response.data.webViewLink
      }
    } catch (error: any) {
      console.error('Google Drive upload error:', error)
      return {
        success: false,
        error: error.message
      }
    }
  }

  async createFolder(folderName: string, parentFolderId?: string): Promise<{ success: boolean; folderId?: string; error?: string }> {
    if (!this.drive) {
      return {
        success: false,
        error: 'Google Drive not configured'
      }
    }

    try {
      const fileMetadata = {
        name: folderName,
        mimeType: 'application/vnd.google-apps.folder',
        parents: parentFolderId ? [parentFolderId] : [GOOGLE_DRIVE_FOLDER_ID!]
      }

      const response = await this.drive.files.create({
        resource: fileMetadata,
        fields: 'id'
      })

      // Make folder publicly accessible
      await this.drive.permissions.create({
        fileId: response.data.id,
        requestBody: {
          role: 'reader',
          type: 'anyone'
        }
      })

      return {
        success: true,
        folderId: response.data.id
      }
    } catch (error: any) {
      console.error('Google Drive create folder error:', error)
      return {
        success: false,
        error: error.message
      }
    }
  }

  async deleteFile(fileId: string): Promise<{ success: boolean; error?: string }> {
    if (!this.drive) {
      return {
        success: false,
        error: 'Google Drive not configured'
      }
    }

    try {
      await this.drive.files.delete({
        fileId: fileId
      })

      return { success: true }
    } catch (error: any) {
      console.error('Google Drive delete error:', error)
      return {
        success: false,
        error: error.message
      }
    }
  }

  async getFileInfo(fileId: string): Promise<{ success: boolean; size?: number; error?: string }> {
    if (!this.drive) {
      return {
        success: false,
        error: 'Google Drive not configured'
      }
    }

    try {
      const response = await this.drive.files.get({
        fileId: fileId,
        fields: 'size'
      })

      return {
        success: true,
        size: parseInt(response.data.size)
      }
    } catch (error: any) {
      console.error('Google Drive get info error:', error)
      return {
        success: false,
        error: error.message
      }
    }
  }
}

export const googleDriveService = new GoogleDriveService()