
import puppeteer from 'puppeteer-extra';
import StealthPlugin from 'puppeteer-extra-plugin-stealth';

// Add stealth plugin to avoid detection
puppeteer.use(StealthPlugin());

export async function getInstagramViews(url: string): Promise<number> {
    let browser;
    try {
        // console.log(`[InstaBrowser] Launching browser for: ${url}`);
        
        browser = await puppeteer.launch({
            headless: true, // Set to false for debugging
            args: [
                '--no-sandbox',
                '--disable-setuid-sandbox',
                '--disable-dev-shm-usage',
                '--disable-accelerated-2d-canvas',
                '--no-first-run',
                '--no-zygote',
                '--single-process', // <- this one doesn't works in Windows
                '--disable-gpu'
            ]
        });

        const page = await browser.newPage();
        
        // Set a realistic viewport
        await page.setViewport({ width: 1280, height: 800 });

        // Navigate to the page
        // console.log('[InstaBrowser] Navigating...');
        await page.goto(url, { waitUntil: 'networkidle2', timeout: 30000 });

        // Wait for the view count element to appear
        // Instagram view counts usually appear in a span or button text
        // We'll look for common patterns
        // console.log('[InstaBrowser] Waiting for content...');
        
        // Try to find the view count in the text content
        // This is a bit generic but robust against class name changes
        const viewCount = await page.evaluate(() => {
            // Helper to parse "1.2M", "10,5 k", "5432"
            const parseCount = (text: string) => {
                text = text.toLowerCase().replace(/views|vues|reproducciones/g, '').trim();
                let multiplier = 1;
                if (text.includes('k')) multiplier = 1000;
                if (text.includes('m')) multiplier = 1000000;
                const num = parseFloat(text.replace(/,/g, '.').replace(/[km]/g, ''));
                return Math.floor(num * multiplier);
            };

            // Strategy 1: Look for text containing "views" or "vues"
            const elements = Array.from(document.querySelectorAll('span, div, a'));
            for (const el of elements) {
                const text = el.textContent || '';
                if (text.match(/([0-9,.]+[KMB]?)\s*(views|vues|reproducciones)/i)) {
                    const match = text.match(/([0-9,.]+[KMB]?)\s*(views|vues|reproducciones)/i);
                    if (match) return parseCount(match[1]);
                }
            }

            // Strategy 2: Look for specific aria-labels or test-ids if known (Instagram changes these often)
            // Strategy 3: Check for the "Play count" in the structured data script if present in DOM
            
            return 0;
        });

        // console.log(`[InstaBrowser] Found views: ${viewCount}`);
        return viewCount || 0;

    } catch (error: any) {
        console.error(`[InstaBrowser] Error: ${error.message}`);
        return 0;
    } finally {
        if (browser) {
            await browser.close();
        }
    }
}
