/**
 * Instagram GraphQL API Client
 * Uses session cookies from get-session.ts to fetch video stats
 */

import { loadSession, InstagramSession } from './get-session'

const GRAPHQL_URL = 'https://www.instagram.com/graphql/query'

// GraphQL Query IDs (may need periodic updates)
const DOC_IDS = {
    MEDIA: '8845758582119845',  // xdt_shortcode_media - for video/post stats
    PROFILE: '7628322937245822' // web_profile_info - for profile stats
}

interface InstagramVideoStats {
    views: number | null
    likes: number | null
    comments: number | null
    thumbnail: string | null
}

interface InstagramProfileStats {
    followers: number | null
    following: number | null
    posts: number | null
    profilePic: string | null
    username: string | null
}

/**
 * Extract shortcode from Instagram URL
 * Supports: /p/, /reel/, /reels/, /tv/
 */
export function extractShortcode(url: string): string | null {
    const patterns = [
        /instagram\.com\/p\/([^\/\?]+)/,
        /instagram\.com\/reel\/([^\/\?]+)/,
        /instagram\.com\/reels\/([^\/\?]+)/,
        /instagram\.com\/tv\/([^\/\?]+)/,
    ]

    for (const pattern of patterns) {
        const match = url.match(pattern)
        if (match) return match[1]
    }

    return null
}

/**
 * Extract username from Instagram URL
 */
export function extractUsername(url: string): string | null {
    const match = url.match(/instagram\.com\/([^\/\?]+)/)
    if (match) {
        const username = match[1]
        // Filter out known paths
        if (['p', 'reel', 'reels', 'tv', 'stories', 'explore'].includes(username)) {
            return null
        }
        return username
    }
    return null
}

/**
 * Fetch video/post stats using GraphQL API
 */
export async function getInstagramVideoStats(shortcode: string, session?: InstagramSession): Promise<InstagramVideoStats | null> {
    const sess = session || loadSession()

    if (!sess) {
        console.error('❌ No Instagram session available. Run get-session.ts first.')
        return null
    }

    try {
        const params = new URLSearchParams()
        params.append('doc_id', DOC_IDS.MEDIA)
        params.append('variables', JSON.stringify({ shortcode }))

        const response = await fetch(GRAPHQL_URL, {
            method: 'POST',
            headers: {
                ...sess.headers,
                'Content-Type': 'application/x-www-form-urlencoded',
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/121.0.0.0 Safari/537.36',
                'Accept': '*/*',
                'Accept-Language': 'en-US,en;q=0.9',
                'Origin': 'https://www.instagram.com',
                'Referer': `https://www.instagram.com/p/${shortcode}/`
            },
            body: params
        })

        if (!response.ok) {
            console.error(`❌ Instagram API error: ${response.status}`)
            return null
        }

        const json = await response.json() as any
        const media = json.data?.xdt_shortcode_media

        if (!media) {
            console.error('❌ No media data in response')
            return null
        }

        return {
            views: media.video_play_count ?? null,
            likes: media.edge_media_preview_like?.count ?? null,
            comments: media.edge_media_to_parent_comment?.count ?? null,
            thumbnail: media.display_url ?? null
        }

    } catch (error) {
        console.error('❌ Error fetching Instagram stats:', error)
        return null
    }
}

/**
 * Fetch profile stats using GraphQL API
 */
export async function getInstagramProfileStats(username: string, session?: InstagramSession): Promise<InstagramProfileStats | null> {
    const sess = session || loadSession()

    if (!sess) {
        console.error('❌ No Instagram session available. Run get-session.ts first.')
        return null
    }

    try {
        const params = new URLSearchParams()
        params.append('doc_id', DOC_IDS.PROFILE)
        params.append('variables', JSON.stringify({ username }))

        const response = await fetch(GRAPHQL_URL, {
            method: 'POST',
            headers: {
                ...sess.headers,
                'Content-Type': 'application/x-www-form-urlencoded',
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/121.0.0.0 Safari/537.36',
                'Accept': '*/*',
                'Accept-Language': 'en-US,en;q=0.9',
                'Origin': 'https://www.instagram.com',
                'Referer': `https://www.instagram.com/${username}/`
            },
            body: params
        })

        if (!response.ok) {
            console.error(`❌ Instagram API error: ${response.status}`)
            return null
        }

        const json = await response.json() as any
        const user = json.data?.user

        if (!user) {
            console.error('❌ No user data in response')
            return null
        }

        return {
            followers: user.edge_followed_by?.count ?? null,
            following: user.edge_follow?.count ?? null,
            posts: user.edge_owner_to_timeline_media?.count ?? null,
            profilePic: user.profile_pic_url_hd ?? user.profile_pic_url ?? null,
            username: user.username ?? null
        }

    } catch (error) {
        console.error('❌ Error fetching Instagram profile:', error)
        return null
    }
}

/**
 * Get views from an Instagram video URL (convenience function)
 */
export async function getInstagramViews(videoUrl: string): Promise<number | null> {
    const shortcode = extractShortcode(videoUrl)

    if (!shortcode) {
        console.error('❌ Invalid Instagram URL:', videoUrl)
        return null
    }

    const stats = await getInstagramVideoStats(shortcode)
    return stats?.views ?? null
}
