import { db } from "@/lib/db"
import { submissions } from "@/lib/db/schema"
import { eq } from "drizzle-orm"
import { getInstagramVideoStats, extractShortcode } from "./instagram/fetch-stats"
// Re-export subscriber functions to maintain compatibility with existing code
export { getTikTokFollowers, getYouTubeFollowers as getYouTubeSubscribers, getInstagramFollowers } from "./tracking/subscriber-tracker"

export interface SocialStats {
    likes: number
    views: number
    engagementRate: number
}

export class SocialScraper {
    private async fetchHtml(url: string): Promise<string> {
        try {
            const response = await fetch(url, {
                headers: {
                    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                    'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
                    'Accept-Language': 'en-US,en;q=0.5'
                }
            })
            if (!response.ok) {
                throw new Error(`Failed to fetch ${url}: ${response.status} ${response.statusText}`)
            }
            return await response.text()
        } catch (error) {
            console.error(`Error fetching ${url}:`, error)
            throw error
        }
    }

    private calculateEngagement(likes: number, views: number): number {
        if (views === 0) return 0
        return (likes / views) * 100
    }

    private async scrapeTikTok(url: string): Promise<SocialStats | null> {
        try {
            const html = await this.fetchHtml(url)

            const likesMatch = html.match(/"diggCount":(\d+)/)
            const viewsMatch = html.match(/"playCount":(\d+)/)

            if (likesMatch && viewsMatch) {
                const likes = parseInt(likesMatch[1])
                const views = parseInt(viewsMatch[1])
                return {
                    likes,
                    views,
                    engagementRate: this.calculateEngagement(likes, views)
                }
            }

            console.warn(`Could not parse TikTok stats for ${url}`)
            return null
        } catch (error) {
            console.error(`Error scraping TikTok ${url}:`, error)
            return null
        }
    }

    private async scrapeYouTube(url: string): Promise<SocialStats | null> {
        try {
            // Convert Shorts URL to Watch URL as suggested by user for better scraping reliability
            let fetchUrl = url
            if (url.includes('/shorts/')) {
                fetchUrl = url.replace('/shorts/', '/watch?v=')
            }

            const html = await this.fetchHtml(fetchUrl)
            let likes = 0
            let views = 0

            // 1. Try to parse ytInitialData for robust extraction
            const jsonMatch = html.match(/var ytInitialData = ({.*?});/)
            if (jsonMatch) {
                try {
                    const data = JSON.parse(jsonMatch[1])

                    // Navigate deep to find like count in Shorts overlay
                    const overlay = data.overlay?.reelPlayerOverlayRenderer
                    if (overlay) {
                        const buttons = overlay.buttonBar?.reelActionBarViewModel?.buttonViewModels
                        if (buttons && buttons.length > 0) {
                            const likeBtn = buttons[0].likeButtonViewModel
                            const defaultBtn = likeBtn?.toggleButtonViewModel?.toggleButtonViewModel?.defaultButtonViewModel?.buttonViewModel
                            if (defaultBtn?.title) {
                                likes = this.parseCount(defaultBtn.title)
                            }
                        }
                    }

                    // Navigate deep to find like count in Watch page (videoPrimaryInfoRenderer)
                    if (likes === 0 || views === 0) {
                        const contents = data.contents?.twoColumnWatchNextResults?.results?.results?.contents
                        if (contents) {
                            const videoPrimaryInfo = contents.find((c: any) => c.videoPrimaryInfoRenderer)?.videoPrimaryInfoRenderer
                            if (videoPrimaryInfo) {
                                // Extract Likes
                                if (likes === 0) {
                                    const topLevelButtons = videoPrimaryInfo.videoActions?.menuRenderer?.topLevelButtons
                                    if (topLevelButtons) {
                                        const likeButton = topLevelButtons.find((b: any) => b.segmentedLikeDislikeButtonViewModel)?.segmentedLikeDislikeButtonViewModel?.likeButtonViewModel?.likeButtonViewModel?.toggleButtonViewModel?.toggleButtonViewModel?.defaultButtonViewModel?.buttonViewModel
                                        if (likeButton?.title) {
                                            likes = this.parseCount(likeButton.title)
                                        }
                                    }
                                }

                                // Extract Precise Views
                                if (views === 0) {
                                    const viewCountText = videoPrimaryInfo.viewCount?.videoViewCountRenderer?.viewCount?.simpleText
                                    if (viewCountText) {
                                        views = this.parseCount(viewCountText)
                                    }
                                }
                            }
                        }
                    }
                } catch (e) {
                    console.error("Error parsing ytInitialData:", e)
                }
            }

            // 2. Fallback/Parallel Regex for Views
            if (views === 0) {
                const viewCountMatch = html.match(/"viewCount":"(\d+)"/)
                if (viewCountMatch) {
                    views = parseInt(viewCountMatch[1])
                }
            }

            if (views === 0) {
                const shortViewCountMatch = html.match(/"shortViewCount":\{"simpleText":"([^"]+)"\}/)
                if (shortViewCountMatch) {
                    views = this.parseCount(shortViewCountMatch[1])
                }
            }

            if (views === 0) {
                const originalViewCountMatch = html.match(/"originalViewCount":"(\d+)"/)
                if (originalViewCountMatch) {
                    views = parseInt(originalViewCountMatch[1])
                }
            }

            // 3. Fallback Regex for Likes if JSON failed
            if (likes === 0) {
                const likeLabelMatch = html.match(/"accessibilityData":{"label":"(\d+) likes"/)
                if (likeLabelMatch) {
                    likes = parseInt(likeLabelMatch[1])
                }
            }

            if (views > 0 || likes > 0) {
                return {
                    likes,
                    views,
                    engagementRate: this.calculateEngagement(likes, views)
                }
            }

            console.warn(`Could not parse YouTube stats for ${url}`)
            return null
        } catch (error) {
            console.error(`Error scraping YouTube ${url}:`, error)
            return null
        }
    }

    private parseCount(str: string): number {
        if (!str) return 0
        str = str.toUpperCase().replace(/\s/g, '')
        str = str.replace(/VIEWS|LIKES|VUES|J'AIME|AUFRUFE/g, '')

        let multiplier = 1
        if (str.includes('K')) {
            multiplier = 1000
            str = str.replace('K', '')
            // If it has a K, it might use dot or comma as decimal separator. 
            // "1.2K" -> 1.2 * 1000 = 1200
            // "1,2K" -> 1.2 * 1000 = 1200
            str = str.replace(',', '.')
        } else if (str.includes('M')) {
            multiplier = 1000000
            str = str.replace('M', '')
            str = str.replace(',', '.')
        } else if (str.includes('B')) {
            multiplier = 1000000000
            str = str.replace('B', '')
            str = str.replace(',', '.')
        } else {
            // No suffix, likely a pure number.
            // "20,672" -> 20672
            // "20.672" -> 20672
            // Remove all non-digits
            str = str.replace(/[^\d]/g, '')
        }

        const val = parseFloat(str)
        return isNaN(val) ? 0 : Math.floor(val * multiplier)
    }

    private async scrapeInstagram(url: string): Promise<SocialStats | null> {
        try {
            // Use the existing robust GraphQL scraper
            const shortcode = extractShortcode(url)
            if (shortcode) {
                const stats = await getInstagramVideoStats(shortcode)
                if (stats) {
                    const likes = stats.likes || 0
                    const views = stats.views || 0
                    return {
                        likes,
                        views,
                        engagementRate: this.calculateEngagement(likes, views)
                    }
                }
            }

            // Fallback to HTML scraping if GraphQL fails (e.g. no session)
            const html = await this.fetchHtml(url)
            let likes = 0

            const ogDescMatch = html.match(/<meta property="og:description" content="([^"]+)"/)
            const metaDescMatch = html.match(/<meta name="description" content="([^"]+)"/)

            const desc = ogDescMatch?.[1] || metaDescMatch?.[1]

            if (desc) {
                const likesMatch = desc.match(/([\d,.]+[KkMm]?)\s*(?:likes?|j'aime)/i)
                if (likesMatch) {
                    likes = this.parseCount(likesMatch[1])
                }
            }

            if (likes > 0) {
                return {
                    likes,
                    views: 0,
                    engagementRate: 0
                }
            }

            console.warn(`Could not parse Instagram stats for ${url} (Login wall likely or GraphQL failed)`)
            return {
                likes: 0,
                views: 0,
                engagementRate: 0
            }
        } catch (error) {
            console.error(`Error scraping Instagram ${url}:`, error)
            return null
        }
    }

    public async getStats(platform: string, url: string) {
        switch (platform.toLowerCase()) {
            case 'tiktok':
                return this.scrapeTikTok(url)
            case 'youtube':
                return this.scrapeYouTube(url)
            case 'instagram':
                return this.scrapeInstagram(url)
            default:
                return null
        }
    }

    // Static method to match the interface expected by benchmark-scrapers.ts
    // This restores the lost functionality
    public static async getVideoStats(url: string): Promise<SocialStats | null> {
        const scraper = new SocialScraper()
        let platform = ''
        if (url.includes('tiktok.com')) platform = 'tiktok'
        else if (url.includes('youtube.com') || url.includes('youtu.be')) platform = 'youtube'
        else if (url.includes('instagram.com')) platform = 'instagram'

        if (!platform) return null
        return scraper.getStats(platform, url)
    }
}

// Export standalone function for compatibility
export async function getVideoStats(url: string) {
    return SocialScraper.getVideoStats(url)
}
