import { db } from "@/lib/db"
import { campaigns } from "@/lib/db/schema"
import { eq, sql } from "drizzle-orm"
import { S3Client, PutObjectCommand } from "@aws-sdk/client-s3"
import { randomUUID } from "crypto"
import path from "path"
import fs from "fs/promises"

const MAX_STORAGE_BYTES = 5 * 1024 * 1024 * 1024 // 5GB

// S3 Client Configuration
const s3Client = new S3Client({
    region: process.env.S3_REGION || "auto",
    endpoint: process.env.S3_ENDPOINT,
    credentials: {
        accessKeyId: process.env.S3_ACCESS_KEY_ID || "",
        secretAccessKey: process.env.S3_SECRET_ACCESS_KEY || "",
    },
})

const BUCKET_NAME = process.env.S3_BUCKET_NAME || "swiplay-assets"
const PUBLIC_URL_BASE = process.env.S3_PUBLIC_URL || ""

export async function saveFile(
    file: File,
    campaignId: string,
    originalName: string
): Promise<{ path: string; size: number; error?: string }> {
    try {
        const fileSize = file.size

        // 1. Check Campaign Quota (if not general)
        if (campaignId !== 'general') {
            const campaign = await db.query.campaigns.findFirst({
                where: eq(campaigns.id, campaignId)
            })

            if (!campaign) return { path: '', size: 0, error: 'Campagne introuvable' }

            const currentUsage = Number(campaign.totalStorageUsed)

            if (currentUsage + fileSize > MAX_STORAGE_BYTES) {
                return { path: '', size: 0, error: 'Quota de stockage dépassé (5GB max)' }
            }
        }

        // 2. Prepare File Info
        const ext = path.extname(originalName)
        const fileName = `${randomUUID()}${ext}`
        const subDir = campaignId === 'general' ? 'general' : `campaigns/${campaignId}`
        const key = `${subDir}/${fileName}`
        const buffer = Buffer.from(await file.arrayBuffer())

        let fileUrl = ""

        // 3. Upload to S3 (or fallback to local if no env vars)
        if (process.env.S3_ENDPOINT && process.env.S3_ACCESS_KEY_ID) {
            await s3Client.send(new PutObjectCommand({
                Bucket: BUCKET_NAME,
                Key: key,
                Body: buffer,
                ContentType: file.type,
                ACL: "public-read", // Adjust based on provider (R2 doesn't support ACLs usually)
            }))
            fileUrl = `${PUBLIC_URL_BASE}/${key}`
        } else {
            // Fallback to local storage for dev without keys
            console.warn("S3 credentials missing, falling back to local storage")
            const UPLOAD_DIR = path.join(process.cwd(), "public", "uploads")
            const uploadDir = path.join(UPLOAD_DIR, subDir)
            await fs.mkdir(uploadDir, { recursive: true })
            const filePath = path.join(uploadDir, fileName)
            await fs.writeFile(filePath, buffer)
            fileUrl = `/uploads/${subDir}/${fileName}`
        }

        // 4. Update DB (Quota) - Only for campaigns
        if (campaignId !== 'general') {
            await db.transaction(async (tx: any) => {
                await tx.update(campaigns)
                    .set({
                        totalStorageUsed: sql`${campaigns.totalStorageUsed} + ${fileSize}`
                    })
                    .where(eq(campaigns.id, campaignId))
            })
        }

        return { path: fileUrl, size: fileSize }
    } catch (error) {
        console.error("Storage Error:", error)
        return { path: '', size: 0, error: 'Erreur interne lors de la sauvegarde' }
    }
}

export async function getCampaignStorageUsage(campaignId: string) {
    const campaign = await db.query.campaigns.findFirst({
        where: eq(campaigns.id, campaignId),
        columns: {
            totalStorageUsed: true,
            storageLimit: true
        }
    })
    return campaign
}
