import axios from 'axios'
import fs from 'fs/promises'
import path from 'path'
import { fileURLToPath } from 'url'

const __filename = fileURLToPath(import.meta.url)
const __dirname = path.dirname(__filename)

// Configuration des créateurs (sera étendue dynamiquement)
const CREATORS_CONFIG: Record<string, any> = {}

// Headers optimisés pour le scraping
const HEADERS = {
  youtube: {
    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
    'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
    'Accept-Language': 'en-US,en;q=0.9',
    'Accept-Encoding': 'gzip, deflate, br',
    'DNT': '1',
    'Connection': 'keep-alive',
    'Upgrade-Insecure-Requests': '1'
  },
  common: {
    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
  }
}

// Fonction de logging
async function log(message: string, level: string = 'INFO') {
  const timestamp = new Date().toISOString()
  const logMessage = `[${timestamp}] ${level}: ${message}`
  // console.log(logMessage)

  try {
    const logDir = path.join(process.cwd(), 'logs')
    await fs.mkdir(logDir, { recursive: true })
    const logFile = path.join(logDir, 'subscriber-tracker.log')
    await fs.appendFile(logFile, logMessage + '\n')
  } catch (error) {
    console.error('Erreur lors de l\'écriture du log:', error)
  }
}

// Fonction pour formater les nombres
export function formatNumber(num: number): string {
  if (num >= 1000000) {
    return (num / 1000000).toFixed(1) + 'M'
  } else if (num >= 1000) {
    return (num / 1000).toFixed(1) + 'k'
  }
  return num.toString()
}

// Fonction pour parser les nombres avec unités
function parseNumber(text: string): number {
  if (!text) return 0

  const cleaned = text.replace(/[,\s]/g, '').toUpperCase()
  const number = parseFloat(cleaned)

  if (cleaned.includes('K')) {
    return Math.round(number * 1000)
  } else if (cleaned.includes('M')) {
    return Math.round(number * 1000000)
  } else if (cleaned.includes('B')) {
    return Math.round(number * 1000000000)
  }

  return Math.round(number) || 0
}

// Fonction pour récupérer les abonnés YouTube (TEMPS RÉEL FONCTIONNEL)
export async function getYouTubeFollowers(channelId: string) {
  try {
    await log(`Récupération YouTube pour ${channelId}...`)

    const cleanChannelId = channelId.replace('@', '')
    const channelUrl = `https://www.youtube.com/@${cleanChannelId}`

    const response = await axios.get(channelUrl, {
      headers: HEADERS.youtube,
      timeout: 20000
    })

    const patterns = [
      /"subscriberCountText":\s*\{\s*"accessibility":\s*\{\s*"accessibilityData":\s*\{\s*"label":\s*"([^"]*subscribers?)"/i,
      /"simpleText":\s*"([0-9,\.KMB]+)\s*subscribers?"/i,
      /"text":\s*"([0-9,\.KMB]+)\s*subscribers?"/i,
      /([0-9,\.KMB]+)\s*subscribers/i
    ]

    for (const pattern of patterns) {
      const match = response.data.match(pattern)
      if (match) {
        let subscriberText = match[1].replace(/\s*subscribers?.*$/i, '').trim()
        const subscribers = parseNumber(subscriberText)

        if (subscribers > 0) {
          await log(`YouTube ${channelId}: ${subscribers} abonnés (temps réel)`)
          return {
            success: true,
            followers: subscribers,
            formatted: formatNumber(subscribers),
            method: 'youtube_realtime',
            confidence: 0.95
          }
        }
      }
    }

    throw new Error('Aucun pattern YouTube trouvé')

  } catch (error: any) {
    await log(`Erreur YouTube ${channelId}: ${error.message}`, 'ERROR')
    return {
      success: false,
      error: error.message
    }
  }
}

// Fonction pour récupérer les abonnés TikTok (TEMPS RÉEL FONCTIONNEL)
export async function getTikTokFollowers(username: string) {
  try {
    await log(`Récupération TikTok pour @${username}...`)

    const tiktokUrl = `https://www.tiktok.com/@${username}`

    const response = await axios.get(tiktokUrl, {
      headers: {
        ...HEADERS.common,
        'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
        'Accept-Language': 'en-US,en;q=0.5'
      },
      timeout: 15000
    })

    const patterns = [
      /"followerCount":(\d+)/,
      /"stats":\{"followerCount":(\d+)/,
      /Followers<\/strong><strong[^>]*>([^<]+)/i
    ]

    for (const pattern of patterns) {
      const match = response.data.match(pattern)
      if (match) {
        const followers = pattern.source.includes('followerCount')
          ? parseInt(match[1])
          : parseNumber(match[1])

        await log(`TikTok @${username}: ${followers} abonnés (temps réel)`)
        return {
          success: true,
          followers: followers,
          formatted: formatNumber(followers),
          method: 'tiktok_realtime',
          confidence: 0.90
        }
      }
    }

    throw new Error('Aucun pattern TikTok trouvé')

  } catch (error: any) {
    await log(`Erreur TikTok @${username}: ${error.message}`, 'ERROR')
    return {
      success: false,
      error: error.message
    }
  }
}

// Fonction pour récupérer les abonnés Instagram (solution réaliste)
// Fonction pour récupérer les abonnés Instagram (Méthode EMBED)
export async function getInstagramFollowers(username: string) {
  try {
    await log(`Instagram @${username}: Récupération via EMBED...`)

    const embedUrl = `https://www.instagram.com/${username}/embed/`

    const response = await axios.get(embedUrl, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,*/*;q=0.8',
        'Accept-Language': 'en-US,en;q=0.5',
        'Accept-Encoding': 'gzip, deflate, br',
        'Referer': 'https://www.instagram.com/',
        'Connection': 'keep-alive',
        'Upgrade-Insecure-Requests': '1',
        'Sec-Fetch-Dest': 'iframe',
        'Sec-Fetch-Mode': 'navigate',
        'Sec-Fetch-Site': 'same-origin'
      },
      timeout: 15000
    })

    if (response.status === 200) {
      const html = response.data

      // Patterns exacts
      const patterns = [
        {
          name: "Context Followers Count",
          regex: /\\"followers_count\\":(\d+)/g,
          priority: 1
        },
        {
          name: "Edge Followed By Count",
          regex: /\\"edge_followed_by\\":\s*\{\s*\\"count\\":(\d+)\s*\}/g,
          priority: 2
        }
      ]

      for (const pattern of patterns) {
        const matches = [...html.matchAll(pattern.regex)]

        if (matches.length > 0) {
          const followers = matches.map((match: any) => parseInt(match[1]))
          const uniqueFollowers = [...new Set(followers)]

          if (uniqueFollowers.length === 1) {
            const count = uniqueFollowers[0] as number
            await log(`Instagram @${username}: ${count} followers (réel)`)
            return {
              success: true,
              followers: count,
              formatted: formatNumber(count),
              method: 'instagram_embed',
              confidence: 1.0
            }
          }
        }
      }
    }

    throw new Error('Aucun pattern trouvé')

  } catch (error: any) {
    await log(`Instagram @${username}: Erreur - ${error.message}`, 'ERROR')
    return {
      success: false,
      error: error.message
    }
  }
}

// Fonction principale pour récupérer les stats d'un créateur
export async function getCreatorStats(creatorId: string, platforms: Record<string, any>) {
  const results: Record<string, any> = {}

  for (const [platform, config] of Object.entries(platforms)) {
    await log(`  🔍 ${platform}: @${config.username || config.channelId}`)

    let stats: any = { success: false, error: 'Non implémenté' }

    switch (platform) {
      case 'instagram':
        stats = await getInstagramFollowers(config.username)
        break

      case 'youtube':
        stats = await getYouTubeFollowers(config.channelId)
        break

      case 'tiktok':
        stats = await getTikTokFollowers(config.username)
        break
    }

    if (stats.success) {
      results[platform] = {
        username: config.username || config.channelId,
        followers: stats.followers,
        formatted: stats.formatted,
        method: stats.method,
        confidence: stats.confidence,
        lastUpdated: new Date().toISOString(),
        success: true
      }
    } else {
      results[platform] = {
        username: config.username || config.channelId,
        error: stats.error,
        method: 'error',
        lastUpdated: new Date().toISOString(),
        success: false
      }
    }

    // Délai entre plateformes
    await new Promise(resolve => setTimeout(resolve, 3000))
  }

  return results
}
