import NextAuth from 'next-auth'
import { authConfig } from './auth.config'
import { rateLimiter } from '@/lib/rate-limit'
import { NextResponse } from 'next/server'
import createMiddleware from 'next-intl/middleware'
import { routing } from './i18n/routing'

const { auth } = NextAuth(authConfig)
const intlMiddleware = createMiddleware(routing)

const authMiddleware = auth((req) => {
  const { nextUrl } = req
  console.log(`[Middleware] Request: ${nextUrl.pathname}, Auth: ${req.auth?.user?.email}, Role: ${req.auth?.user?.role}`)
  const hostname = req.headers.get('host') || ''

  // Define domains relevant to authMiddleware
  const isAppDomain = hostname.startsWith('app.')
  const isOpsDomain = hostname.startsWith('ops.')

  // Rate Limit for API routes
  if (nextUrl.pathname.startsWith('/api/')) {
    const ip = req.headers.get('x-forwarded-for') || 'unknown'
    const isAllowed = rateLimiter.check(100, ip) // 100 requests per minute per IP

    if (!isAllowed) {
      return new NextResponse('Too Many Requests', { status: 429 })
    }
    return // Let API requests pass through
  }

  // Ops Domain Routing
  if (isOpsDomain) {
    // If on ops domain, redirect root to admin login
    if (nextUrl.pathname === '/' || nextUrl.pathname === '/fr' || nextUrl.pathname === '/en') {
      const locale = nextUrl.pathname.startsWith('/en') ? 'en' : 'fr'
      return NextResponse.redirect(new URL(`/${locale}/auth?admin=true`, nextUrl))
    }
    return intlMiddleware(req)
  }

  if (isAppDomain) {
    // If on app domain, redirect root to login or dashboard
    // Check if user is on root or auth page
    const isRootPage = nextUrl.pathname === '/' || nextUrl.pathname === '/fr' || nextUrl.pathname === '/en'
    const isAuthPage = nextUrl.pathname === '/auth' || nextUrl.pathname === '/fr/auth' || nextUrl.pathname === '/en/auth'

    if (isRootPage || isAuthPage) {
      if (req.auth) {
        // Redirect to role-based dashboard if logged in
        const role = req.auth.user.role
        const locale = nextUrl.pathname.startsWith('/en') ? 'en' : 'fr'

        if (role === 'admin') return NextResponse.redirect(new URL(`/${locale}/ops`, nextUrl))
        if (role === 'studio') return NextResponse.redirect(new URL(`/${locale}/studio/dashboard`, nextUrl))
        return NextResponse.redirect(new URL(`/${locale}/creator/dashboard`, nextUrl))
      } else if (isRootPage) {
        // Redirect to login if on root and not logged in
        const locale = nextUrl.pathname === '/en' ? 'en' : 'fr'
        return NextResponse.redirect(new URL(`/${locale}/auth`, nextUrl))
      }
    }

    // Check if creator has completed onboarding
    if (req.auth?.user?.role === 'creator') {
      const onboardingComplete = req.auth.user.onboardingComplete
      const pathWithoutLocale = nextUrl.pathname.replace(/^\/(en|fr)/, '')

      console.log(`[Middleware] Checking onboarding for ${req.auth.user.email}. Complete: ${onboardingComplete}. Path: ${nextUrl.pathname}`)

      if (!onboardingComplete && !pathWithoutLocale.startsWith('/creator/onboarding') && !nextUrl.pathname.startsWith('/api')) {
        console.log(`[Middleware] Redirecting to onboarding from ${nextUrl.pathname}`)
        const locale = nextUrl.pathname.startsWith('/en') ? 'en' : 'fr'
        return NextResponse.redirect(new URL(`/${locale}/creator/onboarding`, nextUrl))
      }
    }

    return intlMiddleware(req)
  }

  // Fallback (e.g. other subdomains or IP access) - treat as landing or 404?
  // For now, let next-intl handle it (likely landing)
  req.headers.set('x-pathname', nextUrl.pathname)
  return intlMiddleware(req)
})

export default function middleware(req: any) {
  const hostname = req.headers.get('host') || ''
  const nextUrl = req.nextUrl
  const isLandingDomain = hostname === 'beta.swiplay.com' || hostname === 'swiplay.com' || hostname === 'www.swiplay.com'

  // Landing Domain Routing
  if (isLandingDomain) {
    // If on landing domain, prevent access to app routes
    if (
      nextUrl.pathname.startsWith('/creator') ||
      nextUrl.pathname.startsWith('/studio') ||
      nextUrl.pathname.startsWith('/ops') ||
      nextUrl.pathname.startsWith('/auth')
    ) {
      // Redirect to app domain
      const newUrl = new URL(nextUrl.pathname, `https://app.swiplay.com`)
      return NextResponse.redirect(newUrl)
    }
    // Serve Landing Page (handled by next-intl for locale)
    return intlMiddleware(req)
  }

  // For app.swiplay.com and ops.swiplay.com
  // Special case: Skip auth middleware for auth pages to prevent redirect loops
  // Special case: Skip auth middleware for auth and verify pages to prevent redirect loops
  const publicPaths = ['/auth', '/verify']
  const isPublicPath = publicPaths.some(path =>
    nextUrl.pathname === path ||
    nextUrl.pathname === `/en${path}` ||
    nextUrl.pathname === `/fr${path}`
  )

  if (isPublicPath) {
    return intlMiddleware(req)
  }

  return authMiddleware(req, {} as any)
}

export const config = {
  // Matcher ignoring static files
  matcher: ['/((?!api|_next/static|_next/image|favicon.ico|.*\\.(?:png|jpg|jpeg|svg|webp|mp4)$).*)'],
}