"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const constants_1 = require("./constants");
const errors_1 = require("./errors");
const fetch_1 = require("./fetch");
const helpers_1 = require("./helpers");
/**
 *
 * @alpha
 *
 * API class for managing Vector Buckets
 * Provides methods for creating, reading, listing, and deleting vector buckets
 *
 * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
 */
class VectorBucketApi {
    /**
     *
     * @alpha
     *
     * Creates a new VectorBucketApi instance
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param url - The base URL for the storage vectors API
     * @param headers - HTTP headers to include in requests
     * @param fetch - Optional custom fetch implementation
     *
     * @example
     * ```typescript
     * const client = new VectorBucketApi(url, headers)
     * ```
     */
    constructor(url, headers = {}, fetch) {
        this.shouldThrowOnError = false;
        this.url = url.replace(/\/$/, '');
        this.headers = Object.assign(Object.assign({}, constants_1.DEFAULT_HEADERS), headers);
        this.fetch = (0, helpers_1.resolveFetch)(fetch);
    }
    /**
     *
     * @alpha
     *
     * Enable throwing errors instead of returning them in the response
     * When enabled, failed operations will throw instead of returning { data: null, error }
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @returns This instance for method chaining
     * @example
     * ```typescript
     * const client = new VectorBucketApi(url, headers)
     * client.throwOnError()
     * const { data } = await client.createBucket('my-bucket') // throws on error
     * ```
     */
    throwOnError() {
        this.shouldThrowOnError = true;
        return this;
    }
    /**
     *
     * @alpha
     *
     * Creates a new vector bucket
     * Vector buckets are containers for vector indexes and their data
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param vectorBucketName - Unique name for the vector bucket
     * @returns Promise with empty response on success or error
     *
     * @throws {StorageVectorsApiError} With code:
     * - `S3VectorConflictException` if bucket already exists (HTTP 409)
     * - `S3VectorMaxBucketsExceeded` if quota exceeded (HTTP 400)
     * - `InternalError` for server errors (HTTP 500)
     *
     * @example
     * ```typescript
     * const { data, error } = await client.createBucket('embeddings-prod')
     * if (error) {
     *   console.error('Failed to create bucket:', error.message)
     * }
     * ```
     */
    createBucket(vectorBucketName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const data = yield (0, fetch_1.post)(this.fetch, `${this.url}/CreateVectorBucket`, { vectorBucketName }, { headers: this.headers });
                return { data: data || {}, error: null };
            }
            catch (error) {
                if (this.shouldThrowOnError) {
                    throw error;
                }
                if ((0, errors_1.isStorageVectorsError)(error)) {
                    return { data: null, error };
                }
                throw error;
            }
        });
    }
    /**
     *
     * @alpha
     *
     * Retrieves metadata for a specific vector bucket
     * Returns bucket configuration including encryption settings and creation time
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param vectorBucketName - Name of the vector bucket to retrieve
     * @returns Promise with bucket metadata or error
     *
     * @throws {StorageVectorsApiError} With code:
     * - `S3VectorNotFoundException` if bucket doesn't exist (HTTP 404)
     * - `InternalError` for server errors (HTTP 500)
     *
     * @example
     * ```typescript
     * const { data, error } = await client.getBucket('embeddings-prod')
     * if (data) {
     *   console.log('Bucket created at:', new Date(data.vectorBucket.creationTime! * 1000))
     * }
     * ```
     */
    getBucket(vectorBucketName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const data = yield (0, fetch_1.post)(this.fetch, `${this.url}/GetVectorBucket`, { vectorBucketName }, { headers: this.headers });
                return { data, error: null };
            }
            catch (error) {
                if (this.shouldThrowOnError) {
                    throw error;
                }
                if ((0, errors_1.isStorageVectorsError)(error)) {
                    return { data: null, error };
                }
                throw error;
            }
        });
    }
    /**
     *
     * @alpha
     *
     * Lists vector buckets with optional filtering and pagination
     * Supports prefix-based filtering and paginated results
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param options - Listing options
     * @param options.prefix - Filter buckets by name prefix
     * @param options.maxResults - Maximum results per page (default: 100)
     * @param options.nextToken - Pagination token from previous response
     * @returns Promise with list of buckets and pagination token
     *
     * @throws {StorageVectorsApiError} With code:
     * - `InternalError` for server errors (HTTP 500)
     *
     * @example
     * ```typescript
     * // List all buckets with prefix 'prod-'
     * const { data, error } = await client.listBuckets({ prefix: 'prod-' })
     * if (data) {
     *   console.log('Found buckets:', data.buckets.length)
     *   // Fetch next page if available
     *   if (data.nextToken) {
     *     const next = await client.listBuckets({ nextToken: data.nextToken })
     *   }
     * }
     * ```
     */
    listBuckets() {
        return tslib_1.__awaiter(this, arguments, void 0, function* (options = {}) {
            try {
                const data = yield (0, fetch_1.post)(this.fetch, `${this.url}/ListVectorBuckets`, options, {
                    headers: this.headers,
                });
                return { data, error: null };
            }
            catch (error) {
                if (this.shouldThrowOnError) {
                    throw error;
                }
                if ((0, errors_1.isStorageVectorsError)(error)) {
                    return { data: null, error };
                }
                throw error;
            }
        });
    }
    /**
     *
     * @alpha
     *
     * Deletes a vector bucket
     * Bucket must be empty before deletion (all indexes must be removed first)
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param vectorBucketName - Name of the vector bucket to delete
     * @returns Promise with empty response on success or error
     *
     * @throws {StorageVectorsApiError} With code:
     * - `S3VectorBucketNotEmpty` if bucket contains indexes (HTTP 400)
     * - `S3VectorNotFoundException` if bucket doesn't exist (HTTP 404)
     * - `InternalError` for server errors (HTTP 500)
     *
     * @example
     * ```typescript
     * // Delete all indexes first, then delete bucket
     * const { error } = await client.deleteBucket('old-bucket')
     * if (error?.statusCode === 'S3VectorBucketNotEmpty') {
     *   console.error('Must delete all indexes first')
     * }
     * ```
     */
    deleteBucket(vectorBucketName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const data = yield (0, fetch_1.post)(this.fetch, `${this.url}/DeleteVectorBucket`, { vectorBucketName }, { headers: this.headers });
                return { data: data || {}, error: null };
            }
            catch (error) {
                if (this.shouldThrowOnError) {
                    throw error;
                }
                if ((0, errors_1.isStorageVectorsError)(error)) {
                    return { data: null, error };
                }
                throw error;
            }
        });
    }
}
exports.default = VectorBucketApi;
//# sourceMappingURL=VectorBucketApi.js.map