import fs from 'fs/promises';
import path from 'path';
import MessageExtractor from '../extractor/MessageExtractor.js';
import formatters from '../formatters/index.js';
import SourceFileScanner from '../source/SourceFileScanner.js';
import CatalogLocales from './CatalogLocales.js';
import CatalogPersister from './CatalogPersister.js';
import SaveScheduler from './SaveScheduler.js';

class CatalogManager {
  /* The source of truth for which messages are used. */
  messagesByFile = (() => new Map())();

  /* Fast lookup for messages by ID across all files,
   * contains the same messages as `messagesByFile`. */
  messagesById = (() => new Map())();

  /**
   * This potentially also includes outdated ones that were initially available,
   * but are not used anymore. This allows to restore them if they are used again.
   **/
  translationsByTargetLocale = (() => new Map())();
  lastWriteByLocale = (() => new Map())();

  // Cached instances

  constructor(config, opts = {}) {
    this.config = config;
    this.saveScheduler = new SaveScheduler(50);
    this.projectRoot = opts.projectRoot || process.cwd();
    this.isDevelopment = opts.isDevelopment ?? false;
    this.messageExtractor = new MessageExtractor({
      isDevelopment: this.isDevelopment,
      projectRoot: this.projectRoot,
      sourceMap: opts.sourceMap
    });
  }
  async getFormatter() {
    if (this.formatter) {
      return this.formatter;
    } else {
      const FormatterClass = (await formatters[this.config.messages.format]()).default;
      this.formatter = new FormatterClass();
      return this.formatter;
    }
  }
  async getPersister() {
    if (this.persister) {
      return this.persister;
    } else {
      this.persister = new CatalogPersister(this.config.messages.path, await this.getFormatter());
      return this.persister;
    }
  }
  async getCatalogLocales() {
    if (this.catalogLocales) {
      return this.catalogLocales;
    } else {
      const messagesDir = path.join(this.projectRoot, this.config.messages.path);
      const formatter = await this.getFormatter();
      this.catalogLocales = new CatalogLocales({
        messagesDir,
        sourceLocale: this.config.sourceLocale,
        extension: formatter.EXTENSION,
        locales: this.config.messages.locales
      });
      return this.catalogLocales;
    }
  }
  async getTargetLocales() {
    const catalogLocales = await this.getCatalogLocales();
    return catalogLocales.getTargetLocales();
  }
  getSrcPaths() {
    return (Array.isArray(this.config.srcPath) ? this.config.srcPath : [this.config.srcPath]).map(srcPath => path.join(this.projectRoot, srcPath));
  }
  getFileMessages(absoluteFilePath) {
    return this.messagesByFile.get(absoluteFilePath);
  }
  async loadMessages() {
    await this.loadSourceMessages();
    await this.loadTargetMessages();
    if (this.isDevelopment) {
      const catalogLocales = await this.getCatalogLocales();
      catalogLocales.subscribeLocalesChange(this.onLocalesChange);
    }
  }
  async loadSourceMessages() {
    // First hydrate from source locale file to potentially init metadata
    await this.loadLocaleMessages(this.config.sourceLocale);

    // Then extract from all source files
    const sourceFiles = await SourceFileScanner.getSourceFiles(this.getSrcPaths());
    await Promise.all(sourceFiles.map(async filePath => this.extractFileMessages(filePath, await fs.readFile(filePath, 'utf8'))));
  }
  async loadLocaleMessages(locale) {
    const persister = await this.getPersister();
    try {
      const messages = await persister.read(locale);
      const fileTime = await persister.getLastModified(locale);
      this.lastWriteByLocale.set(locale, fileTime);
      return messages;
    } catch {
      return [];
    }
  }
  async loadTargetMessages() {
    const targetLocales = await this.getTargetLocales();
    await Promise.all(targetLocales.map(async locale => {
      this.translationsByTargetLocale.set(locale, new Map());
      const messages = await this.loadLocaleMessages(locale);
      for (const message of messages) {
        const translations = this.translationsByTargetLocale.get(locale);
        translations.set(message.id, message.message);
      }
    }));
  }
  async extractFileMessages(absoluteFilePath, source) {
    const result = await this.messageExtractor.processFileContent(absoluteFilePath, source);
    const prevFileMessages = this.messagesByFile.get(absoluteFilePath);

    // Init with all previous ones
    const idsToRemove = Array.from(prevFileMessages?.keys() ?? []);

    // Replace existing messages with new ones
    const fileMessages = new Map();
    for (let message of result.messages) {
      const prevMessage = this.messagesById.get(message.id);

      // Merge with previous message if it exists
      if (prevMessage) {
        // References: The `message` we receive here will always have one
        // reference, which is the current file. We need to merge this with
        // potentially existing references.
        const references = [...(prevMessage.references ?? [])];
        message.references.forEach(ref => {
          if (!references.some(cur => cur.path === ref.path)) {
            references.push(ref);
          }
        });
        references.sort((referenceA, referenceB) => referenceA.path.localeCompare(referenceB.path));
        message = {
          ...message,
          references
        };

        // Description: In case we have conflicting descriptions, the new one wins.
        if (prevMessage.description && !message.description) {
          message = {
            ...message,
            description: prevMessage.description
          };
        }
      }
      this.messagesById.set(message.id, message);
      fileMessages.set(message.id, message);

      // This message continues to exist in this file
      const index = idsToRemove.indexOf(message.id);
      if (index !== -1) idsToRemove.splice(index, 1);
    }

    // Don't delete IDs still used in other files
    const relativeFilePath = path.relative(this.projectRoot, absoluteFilePath);
    const idsToDelete = idsToRemove.filter(id => {
      const message = this.messagesById.get(id);
      return !message?.references?.some(ref => ref.path !== relativeFilePath);
    });

    // Clean up removed messages from `messagesById`
    idsToDelete.forEach(id => {
      this.messagesById.delete(id);
    });

    // Update the stored messages
    const hasMessages = result.messages.length > 0;
    if (hasMessages) {
      this.messagesByFile.set(absoluteFilePath, fileMessages);
    } else {
      this.messagesByFile.delete(absoluteFilePath);
    }
    const changed = this.haveMessagesChanged(prevFileMessages, fileMessages);
    return {
      ...result,
      changed
    };
  }
  haveMessagesChanged(beforeMessages, afterMessages) {
    // If one exists and the other doesn't, there's a change
    if (!beforeMessages) {
      return afterMessages.size > 0;
    }

    // Different sizes means changes
    if (beforeMessages.size !== afterMessages.size) {
      return true;
    }

    // Check differences in beforeMessages vs afterMessages
    for (const [id, msg1] of beforeMessages) {
      const msg2 = afterMessages.get(id);
      if (!msg2 || !this.areMessagesEqual(msg1, msg2)) {
        return true; // Early exit on first difference
      }
    }
    return false;
  }
  areMessagesEqual(msg1, msg2) {
    return msg1.id === msg2.id && msg1.message === msg2.message && msg1.description === msg2.description && this.areReferencesEqual(msg1.references, msg2.references);
  }
  areReferencesEqual(refs1, refs2) {
    // Both undefined or both empty
    if (!refs1 && !refs2) return true;
    if (!refs1 || !refs2) return false;
    if (refs1.length !== refs2.length) return false;

    // Compare each reference
    for (let i = 0; i < refs1.length; i++) {
      if (refs1[i].path !== refs2[i].path) {
        return false;
      }
    }
    return true;
  }
  async save() {
    return this.saveScheduler.schedule(() => this.saveImpl());
  }
  async saveImpl() {
    const messages = Array.from(this.messagesById.values());
    const persister = await this.getPersister();
    await persister.write(this.config.sourceLocale, messages);
    for (const locale of await this.getTargetLocales()) {
      await this.saveLocale(locale);
    }
    return messages.length;
  }
  async saveLocale(locale) {
    const messages = Array.from(this.messagesById.values());
    const persister = await this.getPersister();

    // Check if file was modified externally
    const lastWriteTime = this.lastWriteByLocale.get(locale);
    const currentFileTime = await persister.getLastModified(locale);

    // If file was modified externally, read and merge
    if (currentFileTime && lastWriteTime && currentFileTime > lastWriteTime) {
      const diskMessages = await persister.read(locale);
      const translations = this.translationsByTargetLocale.get(locale);
      for (const diskMessage of diskMessages) {
        // Disk wins: preserve manual edits
        translations.set(diskMessage.id, diskMessage.message);
      }
    }
    const translations = this.translationsByTargetLocale.get(locale);
    const localeMessages = messages.map(message => ({
      ...message,
      message: translations.get(message.id) || ''
    }));
    await persister.write(locale, localeMessages);

    // Update timestamps
    const newTime = await persister.getLastModified(locale);
    this.lastWriteByLocale.set(locale, newTime);
  }
  onLocalesChange = async params => {
    for (const locale of params.added) {
      const translations = new Map();
      this.translationsByTargetLocale.set(locale, translations);
      const messages = await this.loadLocaleMessages(locale);
      for (const message of messages) {
        translations.set(message.id, message.message);
      }
      await this.saveLocale(locale);
    }
    for (const locale of params.removed) {
      this.translationsByTargetLocale.delete(locale);
      this.lastWriteByLocale.delete(locale);
    }
  };
  destroy() {
    this.saveScheduler.destroy();
    if (this.catalogLocales && this.isDevelopment) {
      this.catalogLocales.unsubscribeLocalesChange(this.onLocalesChange);
    }
  }
}

export { CatalogManager as default };
