import fs from 'fs/promises';
import path from 'path';

class CatalogPersister {
  constructor(messagesPath, formatter) {
    this.messagesPath = messagesPath;
    this.formatter = formatter;
  }
  getFilePath(locale) {
    return path.join(this.messagesPath, locale + this.formatter.EXTENSION);
  }
  async read(locale) {
    const filePath = this.getFilePath(locale);
    const content = await fs.readFile(filePath, 'utf8');
    return this.formatter.parse(content, {
      locale
    });
  }
  async write(locale, messages) {
    const filePath = this.getFilePath(locale);
    const content = this.formatter.serialize(messages, {
      locale
    });
    try {
      const outputDir = path.dirname(filePath);
      await fs.mkdir(outputDir, {
        recursive: true
      });
      await fs.writeFile(filePath, content);
    } catch (error) {
      console.error(`❌ Failed to write catalog: ${error}`);
    }
  }
  async getLastModified(locale) {
    const filePath = this.getFilePath(locale);
    try {
      const stats = await fs.stat(filePath);
      return stats.mtime;
    } catch {
      return undefined;
    }
  }
}

export { CatalogPersister as default };
