/**
 * De-duplicates excessive save invocations,
 * while keeping a single one instant.
 */
class SaveScheduler {
  isSaving = false;
  pendingResolvers = [];
  constructor(delayMs = 50) {
    this.delayMs = delayMs;
  }
  async schedule(saveTask) {
    return new Promise((resolve, reject) => {
      this.pendingResolvers.push({
        resolve,
        reject
      });
      if (this.pendingResolvers.length === 1 && !this.isSaving) {
        // No pending saves and not currently saving, save immediately
        this.executeSave(saveTask);
      } else if (this.pendingResolvers.length > 1) {
        // Multiple pending saves, schedule/reschedule save
        this.scheduleSave(saveTask);
      }
    });
  }
  scheduleSave(saveTask) {
    if (this.saveTimeout) {
      clearTimeout(this.saveTimeout);
    }
    this.saveTimeout = setTimeout(() => {
      this.executeSave(saveTask);
    }, this.delayMs);
  }
  async executeSave(saveTask) {
    if (this.isSaving) {
      return;
    }
    this.isSaving = true;
    try {
      const result = await saveTask();

      // Resolve all pending promises with the same result
      this.pendingResolvers.forEach(({
        resolve
      }) => resolve(result));
    } catch (error) {
      // Reject all pending promises with the same error
      this.pendingResolvers.forEach(({
        reject
      }) => reject(error));
    } finally {
      this.pendingResolvers = [];
      this.isSaving = false;
    }
  }
  destroy() {
    if (this.saveTimeout) {
      clearTimeout(this.saveTimeout);
      this.saveTimeout = undefined;
    }
    this.pendingResolvers = [];
    this.isSaving = false;
  }
}

export { SaveScheduler as default };
