import POParser from 'po-parser';
import { setNestedProperty } from '../utils/ObjectUtils.js';
import Formatter from './Formatter.js';
import { getSortedMessages } from './utils.js';

class POFormatter extends Formatter {
  // See also https://www.gnu.org/software/gettext/manual/html_node/Header-Entry.html
  static DEFAULT_METADATA = {
    // Recommended by spec
    'Content-Type': 'text/plain; charset=utf-8',
    'Content-Transfer-Encoding': '8bit',
    // Otherwise other tools might set this
    'X-Generator': 'next-intl',
    // Crowdin defaults to using msgid as source key
    'X-Crowdin-SourceKey': 'msgstr'
  };
  EXTENSION = '.po';

  // Metadata is stored so it can be retained when writing
  metadataByLocale = (() => new Map())();
  parse(content, context) {
    const catalog = POParser.parse(content);

    // Store metadata for this locale
    if (catalog.meta) {
      this.metadataByLocale.set(context.locale, catalog.meta);
    }
    return catalog.messages || [];
  }
  serialize(messages, context) {
    const meta = {
      Language: context.locale,
      ...POFormatter.DEFAULT_METADATA,
      ...this.metadataByLocale.get(context.locale)
    };
    return POParser.serialize({
      meta,
      messages: getSortedMessages(messages)
    });
  }
  toJSONString(source, context) {
    const parsed = this.parse(source, context);
    const messagesObject = {};
    for (const message of parsed) {
      setNestedProperty(messagesObject, message.id, message.message);
    }
    return JSON.stringify(messagesObject, null, 2);
  }
}

export { POFormatter as default };
