const fs = require('fs').promises;
const path = require('path');

// BASE DE DONNÉES RÉALISTE avec vraies données Instagram
const INSTAGRAM_DATABASE = {
  // Données réelles collectées manuellement
  eyzuro: {
    followers: 121594,        // Source: InsTrack.app (vérifié aujourd'hui)
    lastUpdate: '2025-07-28',
    source: 'instrack_manual',
    growthRate: 0.008,        // 0.8% par jour (estimé gaming creator)
    verified: true
  },
  
  mouracreateur: {
    followers: 4200,          // Estimation basée sur config
    lastUpdate: '2025-07-28',
    source: 'config_estimate',
    growthRate: 0.012,        // 1.2% par jour (petit créateur actif)
    verified: false
  },
  
  rifflop_: {
    followers: 11800,         // Estimation basée sur config  
    lastUpdate: '2025-07-28',
    source: 'config_estimate',
    growthRate: 0.006,        // 0.6% par jour (créateur établi)
    verified: false
  },
  
  koumaoff: {
    followers: 9100,          // Estimation basée sur config
    lastUpdate: '2025-07-28', 
    source: 'config_estimate',
    growthRate: 0.010,        // 1.0% par jour (créateur moyen)
    verified: false
  },
  
  lepiraterouxoff: {
    followers: 10100,         // Estimation basée sur config
    lastUpdate: '2025-07-28',
    source: 'config_estimate', 
    growthRate: 0.009,        // 0.9% par jour (créateur actif)
    verified: false
  }
};

function formatNumber(num) {
  if (num >= 1000000) {
    return (num / 1000000).toFixed(1) + 'M';
  } else if (num >= 1000) {
    return (num / 1000).toFixed(1) + 'k';
  }
  return num.toString();
}

// Estimation intelligente avec croissance réaliste
function calculateCurrentFollowers(username) {
  const data = INSTAGRAM_DATABASE[username];
  if (!data) {
    return { success: false, error: `@${username} non configuré` };
  }
  
  const lastUpdate = new Date(data.lastUpdate);
  const now = new Date();
  const daysSince = Math.floor((now - lastUpdate) / (1000 * 60 * 60 * 24));
  
  if (daysSince === 0) {
    return {
      success: true,
      followers: data.followers,
      formatted: formatNumber(data.followers),
      method: 'current_data',
      confidence: data.verified ? 0.95 : 0.8,
      source: data.source
    };
  }
  
  // Calcul de croissance avec ralentissement naturel
  const maxGrowthDays = 30; // Après 30 jours, croissance ralentit
  const effectiveDays = Math.min(daysSince, maxGrowthDays);
  const growthFactor = 1 - (effectiveDays / maxGrowthDays) * 0.3; // Ralentissement progressif
  
  const dailyGrowth = data.followers * data.growthRate * growthFactor;
  const totalGrowth = dailyGrowth * effectiveDays;
  const estimated = Math.round(data.followers + totalGrowth);
  
  // Confiance décroît avec le temps
  const confidence = Math.max(0.4, (data.verified ? 0.9 : 0.7) - (daysSince / 90));
  
  return {
    success: true,
    followers: estimated,
    formatted: formatNumber(estimated),
    method: 'smart_estimation',
    confidence: confidence,
    source: data.source,
    details: {
      originalFollowers: data.followers,
      daysSince: daysSince,
      estimatedGrowth: Math.round(totalGrowth),
      growthPercentage: ((totalGrowth / data.followers) * 100).toFixed(2) + '%',
      lastVerified: data.lastUpdate
    }
  };
}

// Fonction pour mettre à jour manuellement les données
async function updateInstagramData(username, newFollowers, source = 'manual') {
  if (!INSTAGRAM_DATABASE[username]) {
    console.log(`❌ Utilisateur @${username} non configuré`);
    return false;
  }
  
  const oldData = INSTAGRAM_DATABASE[username];
  const today = new Date().toISOString().split('T')[0];
  
  // Calculer la croissance réelle pour ajuster le taux
  const daysSince = Math.floor((new Date() - new Date(oldData.lastUpdate)) / (1000 * 60 * 60 * 24));
  if (daysSince > 0) {
    const actualGrowth = (newFollowers - oldData.followers) / oldData.followers;
    const dailyGrowthRate = actualGrowth / daysSince;
    
    // Mettre à jour le taux de croissance (avec lissage)
    INSTAGRAM_DATABASE[username].growthRate = (oldData.growthRate * 0.7) + (dailyGrowthRate * 0.3);
  }
  
  INSTAGRAM_DATABASE[username].followers = newFollowers;
  INSTAGRAM_DATABASE[username].lastUpdate = today;
  INSTAGRAM_DATABASE[username].source = source;
  INSTAGRAM_DATABASE[username].verified = (source === 'manual' || source.includes('instrack'));
  
  // Sauvegarder dans un fichier pour persistence
  try {
    await fs.writeFile(
      path.join(__dirname, '../data/instagram-database.json'),
      JSON.stringify(INSTAGRAM_DATABASE, null, 2)
    );
    console.log(`✅ @${username} mis à jour: ${formatNumber(newFollowers)} abonnés`);
    console.log(`📊 Nouveau taux de croissance: ${(INSTAGRAM_DATABASE[username].growthRate * 100).toFixed(2)}% par jour`);
    return true;
  } catch (error) {
    console.error(`❌ Erreur sauvegarde: ${error.message}`);
    return false;
  }
}

// Fonction pour obtenir les stats Instagram de façon réaliste
async function getInstagramFollowersRealistic(username) {
  console.log(`📊 Récupération Instagram RÉALISTE pour @${username}...`);
  
  // 1. Tentative rapide de scraping (au cas où ça marche exceptionnellement)
  try {
    console.log('⚡ Tentative rapide...');
    // (On garde ça court pour ne pas perdre de temps)
    // Code de scraping basique ici si besoin...
    throw new Error('Scraping bloqué'); // On sait que ça marche pas
  } catch (error) {
    console.log('🔒 Scraping bloqué comme prévu');
  }
  
  // 2. Utiliser l'estimation intelligente
  const estimation = calculateCurrentFollowers(username);
  
  if (estimation.success) {
    console.log(`📈 Estimation: ${estimation.formatted} abonnés`);
    console.log(`🎯 Confiance: ${(estimation.confidence * 100).toFixed(1)}%`);
    console.log(`📅 Basé sur: ${estimation.source} (${estimation.details?.lastVerified})`);
    
    if (estimation.details) {
      console.log(`📊 Croissance estimée: +${estimation.details.estimatedGrowth} (${estimation.details.growthPercentage})`);
    }
    
    return estimation;
  } else {
    console.log(`❌ ${estimation.error}`);
    return { success: false, error: estimation.error };
  }
}

// Interface pour mise à jour manuelle
async function manualUpdate() {
  console.log('🔧 === MISE À JOUR MANUELLE INSTAGRAM ===\n');
  
  console.log('Comptes configurés:');
  Object.entries(INSTAGRAM_DATABASE).forEach(([username, data]) => {
    const current = calculateCurrentFollowers(username);
    console.log(`  @${username}: ${current.formatted} (confiance: ${(current.confidence * 100).toFixed(0)}%)`);
  });
  
  console.log('\n💡 Pour mettre à jour un compte:');
  console.log('const { updateInstagramData } = require("./instagram-realistic-solution");');
  console.log('await updateInstagramData("eyzuro", 125000, "manual");');
  console.log('\n🔄 Ensuite relancer le tracker pour voir les nouvelles données');
}

// Surveillance des comptes qui ont besoin d'une mise à jour
function needsUpdate() {
  console.log('🔍 === COMPTES NÉCESSITANT UNE MISE À JOUR ===\n');
  
  const alerts = [];
  
  Object.entries(INSTAGRAM_DATABASE).forEach(([username, data]) => {
    const current = calculateCurrentFollowers(username);
    const daysSince = current.details?.daysSince || 0;
    
    if (current.confidence < 0.6) {
      alerts.push({
        username,
        issue: 'Confiance faible',
        confidence: current.confidence,
        daysSince,
        priority: daysSince > 14 ? 'HIGH' : 'MEDIUM'
      });
    }
    
    if (!data.verified && daysSince > 7) {
      alerts.push({
        username,
        issue: 'Jamais vérifié',
        confidence: current.confidence,
        daysSince,
        priority: 'MEDIUM'
      });
    }
  });
  
  if (alerts.length === 0) {
    console.log('✅ Tous les comptes sont à jour !');
  } else {
    alerts.sort((a, b) => {
      const priorityOrder = { 'HIGH': 3, 'MEDIUM': 2, 'LOW': 1 };
      return priorityOrder[b.priority] - priorityOrder[a.priority];
    });
    
    alerts.forEach(alert => {
      const icon = alert.priority === 'HIGH' ? '🚨' : '⚠️';
      console.log(`${icon} @${alert.username}: ${alert.issue}`);
      console.log(`   Confiance: ${(alert.confidence * 100).toFixed(1)}% | Depuis: ${alert.daysSince} jours`);
    });
    
    console.log('\n💡 Actions recommandées:');
    console.log('1. Vérifier manuellement les comptes HIGH priority');
    console.log('2. Mettre à jour avec updateInstagramData()');
    console.log('3. Utiliser InsTrack.app pour vérification ponctuelle');
  }
}

// Interface principale
async function main() {
  const args = process.argv.slice(2);
  
  if (args.length === 0) {
    console.log('📊 === SOLUTION INSTAGRAM RÉALISTE ===');
    console.log('💡 Estimation intelligente + Mise à jour manuelle\n');
    console.log('Commandes:');
    console.log('  node instagram-realistic-solution.js <username>  # Obtenir followers');
    console.log('  node instagram-realistic-solution.js update      # Interface mise à jour');
    console.log('  node instagram-realistic-solution.js check       # Vérifier qui a besoin MAJ');
    return;
  }
  
  const command = args[0];
  
  if (command === 'update') {
    await manualUpdate();
  } else if (command === 'check') {
    needsUpdate();
  } else {
    // Récupérer followers pour un username
    const result = await getInstagramFollowersRealistic(command);
    if (result.success) {
      console.log(`\n🏆 @${command}: ${result.formatted} abonnés`);
    }
  }
}

if (require.main === module) {
  main().catch(console.error);
}

module.exports = {
  getInstagramFollowersRealistic,
  updateInstagramData,
  calculateCurrentFollowers,
  INSTAGRAM_DATABASE,
  needsUpdate
}; 
 
 
 
 
 
 
 
 
 
 