
import fs from 'fs';
import path from 'path';

const ROOT_DIR = path.resolve(__dirname, '..');
const TRASH_DIR = path.join(ROOT_DIR, 'trash');
const UNUSED_FILE = path.join(ROOT_DIR, 'unused_files.json');

const WHITELIST_PREFIXES = [
    '.gitignore',
    'drizzle/meta',
    'messages',
    'public/uploads',
    'scripts',
    'tests',
    'types',
    'node_modules', // Should not be in list but just in case
];

async function main() {
    if (!fs.existsSync(UNUSED_FILE)) {
        console.error('unused_files.json not found');
        process.exit(1);
    }

    const unusedFiles: string[] = JSON.parse(fs.readFileSync(UNUSED_FILE, 'utf-8'));

    const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
    const trashBatchDir = path.join(TRASH_DIR, timestamp);

    if (!fs.existsSync(trashBatchDir)) {
        fs.mkdirSync(trashBatchDir, { recursive: true });
    }

    const restoreMap: Record<string, string> = {};
    let movedCount = 0;

    for (const fileRel of unusedFiles) {
        // Check whitelist
        if (WHITELIST_PREFIXES.some(prefix => fileRel.startsWith(prefix))) {
            console.log(`Skipping whitelisted: ${fileRel}`);
            continue;
        }

        const sourcePath = path.join(ROOT_DIR, fileRel);
        const destPath = path.join(trashBatchDir, fileRel);

        if (fs.existsSync(sourcePath)) {
            const destDir = path.dirname(destPath);
            if (!fs.existsSync(destDir)) {
                fs.mkdirSync(destDir, { recursive: true });
            }

            fs.renameSync(sourcePath, destPath);
            restoreMap[fileRel] = destPath; // Map original relative path to new absolute path (or relative to trash)
            console.log(`Moved: ${fileRel}`);
            movedCount++;
        } else {
            console.warn(`File not found: ${fileRel}`);
        }
    }

    fs.writeFileSync(
        path.join(trashBatchDir, 'restore_map.json'),
        JSON.stringify(restoreMap, null, 2)
    );

    console.log(`\nMoved ${movedCount} files to ${trashBatchDir}`);
    console.log(`Restore map saved to ${path.join(trashBatchDir, 'restore_map.json')}`);
}

main().catch(console.error);
