import 'dotenv/config'
import { db } from '@/lib/db/index'
import { users, campaigns, submissions, campaignApplications } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { v4 as uuidv4 } from 'uuid'

const API_URL = 'http://localhost:3000/api'

async function main() {
    console.log('🚀 Starting API Test Suite...')

    // 1. Setup Test Users
    console.log('\n1️⃣  Setting up Test Users...')

    const studioId = uuidv4()
    const creatorId = uuidv4()

    try {
        // Create Studio
        await db.insert(users).values({
            id: studioId,
            email: `studio-${studioId.substring(0, 8)}@test.com`,
            name: 'Test Studio',
            role: 'studio',
            password: 'hashed_password',
            isApproved: 1,
            emailVerified: new Date()
        })
        console.log('✅ Created Studio:', studioId)

        // Create Creator
        await db.insert(users).values({
            id: creatorId,
            email: `creator-${creatorId.substring(0, 8)}@test.com`,
            name: 'Test Creator',
            role: 'creator',
            password: 'hashed_password',
            isApproved: 1,
            onboardingComplete: 1,
            emailVerified: new Date(),
            balance: '0'
        })
        console.log('✅ Created Creator:', creatorId)

    } catch (e) {
        console.error('❌ Failed to create users:', e)
        process.exit(1)
    }

    // 2. Create Campaign (Studio)
    console.log('\n2️⃣  Testing POST /api/campaigns (Create Campaign)...')
    let campaignId = ''

    try {
        const res = await fetch(`${API_URL}/campaigns`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-Test-User-Id': studioId,
                'X-Forwarded-For': `10.0.${Math.floor(Math.random() * 255)}.${Math.floor(Math.random() * 255)}`
            },
            body: JSON.stringify({
                title: 'Test Campaign API',
                description: 'Testing the API flow',
                budget: 1000,
                pricePerView: 0.01,
                creatorCap: 50,
                platform: 'tiktok',
                tags: ['Test', 'API'],
                coverImage: 'https://example.com/cover.jpg',
                images: [],
                videos: []
            })
        })

        if (!res.ok) throw new Error(`Status ${res.status}: ${await res.text()}`)
        const data = await res.json()
        campaignId = data.campaign.id
        console.log('✅ Campaign Created:', campaignId)

        // Publish Campaign
        console.log('   Publishing Campaign...')
        const pubRes = await fetch(`${API_URL}/campaigns/${campaignId}`, {
            method: 'PATCH',
            headers: {
                'Content-Type': 'application/json',
                'X-Test-User-Id': studioId
            },
            body: JSON.stringify({ status: 'active' })
        })
        if (!pubRes.ok) throw new Error(`Publish Failed: ${await pubRes.text()}`)
        console.log('✅ Campaign Published')

    } catch (e) {
        console.error('❌ Campaign Creation Failed:', e)
        process.exit(1)
    }

    // 3. Apply to Campaign (Creator)
    console.log('\n3️⃣  Testing POST /api/creator/campaigns/[id]/apply...')
    let applicationId = ''

    try {
        const res = await fetch(`${API_URL}/creator/campaigns/${campaignId}/apply`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-Test-User-Id': creatorId
            },
            body: JSON.stringify({ message: 'I want to join!' })
        })

        if (!res.ok) throw new Error(`Status ${res.status}: ${await res.text()}`)
        const data = await res.json()
        applicationId = data.applicationId // Verify response structure
        console.log('✅ Application Submitted')
    } catch (e) {
        console.error('❌ Application Failed:', e)
        // Continue? No, flow is broken
        process.exit(1)
    }

    // 4. Approve Application (Studio)
    // We need to find the application ID first if not returned, but let's assume we can fetch campaign details
    console.log('\n4️⃣  Testing PATCH /api/applications/[id] (Approve)...')

    try {
        // Fetch campaign to get application ID
        const campRes = await fetch(`${API_URL}/campaigns/${campaignId}`, {
            headers: { 'X-Test-User-Id': studioId }
        })
        const campData = await campRes.json()
        const app = campData.applications.find((a: any) => a.creatorId === creatorId)

        if (!app) throw new Error('Application not found in campaign details')

        const res = await fetch(`${API_URL}/applications/${app.id}`, {
            method: 'PATCH',
            headers: {
                'Content-Type': 'application/json',
                'X-Test-User-Id': studioId
            },
            body: JSON.stringify({ status: 'approved' })
        })

        if (!res.ok) throw new Error(`Status ${res.status}: ${await res.text()}`)
        console.log('✅ Application Approved')
    } catch (e) {
        console.error('❌ Approval Failed:', e)
        process.exit(1)
    }

    // 5. Submit Video (Creator)
    console.log('\n5️⃣  Testing POST /api/creator/campaigns/[id]/submit...')

    try {
        const res = await fetch(`${API_URL}/creator/campaigns/${campaignId}/submit`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-Test-User-Id': creatorId
            },
            body: JSON.stringify({
                videoUrl: 'https://www.tiktok.com/@user/video/1234567890',
                platform: 'tiktok'
            })
        })

        if (!res.ok) throw new Error(`Status ${res.status}: ${await res.text()}`)
        console.log('✅ Video Submitted')
    } catch (e) {
        console.error('❌ Submission Failed:', e)
        process.exit(1)
    }

    // 5.5. Approve Submission (Studio)
    console.log('\n5️⃣.5️⃣  Testing PATCH /api/submissions/[id] (Approve Submission)...')
    try {
        // Fetch submission ID
        // Since we don't have an endpoint to list submissions easily without auth, we can query DB or assume we get it from response?
        // Wait, POST submit returned the submission object!
        // But I didn't capture it in the previous step.
        // Let's update step 5 to capture submission ID.

        // Actually, I'll just query the DB directly here for simplicity since I have db access in this script.
        const sub = await db.query.submissions.findFirst({
            where: eq(submissions.creatorId, creatorId)
        })
        if (!sub) throw new Error('Submission not found in DB')

        const res = await fetch(`${API_URL}/submissions/${sub.id}`, {
            method: 'PATCH',
            headers: {
                'Content-Type': 'application/json',
                'X-Test-User-Id': studioId
            },
            body: JSON.stringify({ status: 'approved' })
        })

        if (!res.ok) throw new Error(`Status ${res.status}: ${await res.text()}`)
        console.log('✅ Submission Approved')

    } catch (e) {
        console.error('❌ Submission Approval Failed:', e)
        process.exit(1)
    }

    // 6. Run Cron (System)
    console.log('\n6️⃣  Testing GET /api/cron/update-views...')

    try {
        const res = await fetch(`${API_URL}/cron/update-views`)
        if (!res.ok) throw new Error(`Status ${res.status}: ${await res.text()}`)
        const data = await res.json()
        console.log('✅ Cron Job Ran:', data)
    } catch (e) {
        console.error('❌ Cron Job Failed:', e)
    }

    console.log('\n🎉 API Test Suite Completed Successfully!')
    process.exit(0)
}

main()
