import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { payoutRequests, users } from '@/lib/db/schema'
import { eq, desc } from 'drizzle-orm'
import { authenticateRequest, requireRole } from '@/lib/auth-middleware'

export async function GET(request: NextRequest) {
    try {
        const authResult = await authenticateRequest(request)
        if (!authResult.success) {
            return NextResponse.json({ error: authResult.error }, { status: 401 })
        }

        const roleCheck = requireRole(authResult.role, 'admin')
        if (!roleCheck.allowed) {
            return NextResponse.json({ error: roleCheck.error }, { status: 403 })
        }

        const requests = await db
            .select({
                id: payoutRequests.id,
                amount: payoutRequests.amount,
                status: payoutRequests.status,
                createdAt: payoutRequests.createdAt,
                creatorId: payoutRequests.creatorId,
                creatorName: users.name,
                creatorEmail: users.email,
                iban: users.iban,
                bic: users.bic,
            })
            .from(payoutRequests)
            .leftJoin(users, eq(payoutRequests.creatorId, users.id))
            .where(eq(payoutRequests.status, 'pending'))
            .orderBy(desc(payoutRequests.createdAt))

        return NextResponse.json(requests)

    } catch (error) {
        console.error('Get payouts error:', error)
        return NextResponse.json({ error: 'Erreur serveur' }, { status: 500 })
    }
}
