import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { db } from '@/lib/db'
import { campaigns, gameKeys as gameKeysTable, campaignAssets } from '@/lib/db/schema'
import { eq, desc } from 'drizzle-orm'
import { campaignSchema } from '@/lib/validations/campaign'
import { campaignRateLimiter } from '@/lib/rate-limit-db'
import { getTranslations } from 'next-intl/server'
import { sendDiscordNotification } from '@/lib/discord'
import logger from '@/lib/logger'

export async function POST(request: NextRequest) {
  const t = await getTranslations('Api.Errors')

  try {
    const ip = request.headers.get('x-forwarded-for') ?? '127.0.0.1'

    if (!await campaignRateLimiter.check(10, ip)) {
      return NextResponse.json({ error: t('rate_limit') }, { status: 429 })
    }

    const session = await auth()
    if (!session || session.user.role !== 'studio') {
      return NextResponse.json({ error: t('unauthorized') }, { status: 401 })
    }

    const body = await request.json()
    const validation = campaignSchema.safeParse(body)

    if (!validation.success) {
      return NextResponse.json({ error: validation.error.errors[0].message }, { status: 400 })
    }

    const {
      title, description, budget, pricePerView, creatorCap, startDate, endDate, platform,
      coverImage, icon, images, tags,
      gameKeys, assets
    } = validation.data

    // Transactional insert
    const newCampaign = await db.transaction(async (tx) => {
      // 1. Create Campaign
      const [campaign] = await tx.insert(campaigns).values({
        studioId: session.user.id,
        title,
        description,
        budget: budget.toString(),
        pricePerView: pricePerView.toString(),
        creatorCap: creatorCap ? creatorCap.toString() : null,
        status: 'draft',
        startDate: startDate ? new Date(startDate) : undefined,
        endDate: endDate ? new Date(endDate) : undefined,
        coverImage,
        icon,
        images: images || [],
        tags: tags || [],
      }).returning()

      // 2. Insert Game Keys
      if (gameKeys && gameKeys.length > 0) {
        for (const group of gameKeys) {
          const keysList = group.keys.split('\n').map(k => k.trim()).filter(k => k.length > 0)
          if (keysList.length > 0) {
            await tx.insert(gameKeysTable).values(
              keysList.map(key => ({
                campaignId: campaign.id,
                platform: group.platform,
                key,
                isClaimed: 0
              }))
            )
          }
        }
      }

      // 3. Insert Assets
      if (assets && assets.length > 0) {
        await tx.insert(campaignAssets).values(
          assets.map(asset => ({
            campaignId: campaign.id,
            fileName: asset.name,
            fileType: asset.type, // 'file' or 'link'
            driveUrl: asset.url,
            fileId: 'manual_upload', // Placeholder
            size: asset.size || 0
          }))
        )
      }

      return campaign
    })

    // Discord Log
    const { sendDiscordNotification } = await import('@/lib/discord')
    await sendDiscordNotification(
      '📢 Nouvelle Campagne Créée',
      `Le studio **${session.user.name || session.user.email}** a créé la campagne **${title}**.`,
      'info',
      [
        { name: 'Studio ID', value: session.user.id, inline: true },
        { name: 'Budget', value: `${budget}€`, inline: true },
        { name: 'PPV', value: `${pricePerView}€`, inline: true }
      ]
    )

    return NextResponse.json({ success: true, campaign: newCampaign })
  } catch (error) {
    logger.error({ err: error }, 'Error creating campaign')
    return NextResponse.json({ error: t('server_error') }, { status: 500 })
  }
}

export async function GET(request: NextRequest) {
  const t = await getTranslations('Api.Errors')

  try {
    const session = await auth()
    if (!session) {
      return NextResponse.json({ error: t('unauthorized') }, { status: 401 })
    }

    // If studio, return their campaigns
    if (session.user.role === 'studio') {
      const studioCampaigns = await db.query.campaigns.findMany({
        where: eq(campaigns.studioId, session.user.id),
        orderBy: [desc(campaigns.createdAt)]
      })
      return NextResponse.json({ campaigns: studioCampaigns })
    }

    // If creator, return active campaigns (marketplace)
    if (session.user.role === 'creator') {
      const activeCampaigns = await db.query.campaigns.findMany({
        where: eq(campaigns.status, 'active'),
        orderBy: [desc(campaigns.createdAt)],
        with: {
          studio: {
            columns: {
              name: true,
              image: true,
              company: true
            }
          }
        }
      })
      return NextResponse.json({ campaigns: activeCampaigns })
    }

    return NextResponse.json({ campaigns: [] })
  } catch (error) {
    logger.error({ err: error }, 'Error fetching campaigns')
    return NextResponse.json({ error: t('server_error') }, { status: 500 })
  }
}