import { auth } from '@/auth'
import { db } from '@/lib/db'
import { users } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { NextResponse } from 'next/server'
import { z } from 'zod'

const eurSchema = z.object({
    currency: z.literal('EUR'),
    iban: z.string().min(15, "IBAN invalide"),
    bic: z.string().min(8, "BIC invalide"),
    fullName: z.string().min(1, "Nom requis"),
    bankName: z.string().optional(),
})

const usdSchema = z.object({
    currency: z.literal('USD'),
    account: z.string().min(4, "Numéro de compte requis"),
    routing: z.string().min(4, "Routing Number requis"),
    fullName: z.string().min(1, "Nom requis"),
    bankName: z.string().min(1, "Nom de la banque requis"),
})

const bankingSchema = z.discriminatedUnion("currency", [eurSchema, usdSchema])

export async function POST(req: Request) {
    try {
        const session = await auth()
        if (!session || !session.user) {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
        }

        const body = await req.json()
        const validated = bankingSchema.parse(body)

        if (validated.currency === 'EUR') {
            await db.update(users)
                .set({
                    currency: 'EUR',
                    iban: validated.iban,
                    bic: validated.bic,
                    bankName: validated.bankName || null,
                    accountNumber: null, // Clear USD fields
                    routingNumber: null,
                })
                .where(eq(users.id, session.user.id))
        } else {
            await db.update(users)
                .set({
                    currency: 'USD',
                    accountNumber: validated.account,
                    routingNumber: validated.routing,
                    bankName: validated.bankName,
                    iban: null, // Clear EUR fields
                    bic: null,
                })
                .where(eq(users.id, session.user.id))
        }

        return NextResponse.json({ success: true })
    } catch (error) {
        console.error('Banking update error:', error)
        if (error instanceof z.ZodError) {
            return NextResponse.json({ error: error.errors[0].message }, { status: 400 })
        }
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}
