import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { db } from '@/lib/db'
import { users, creatorProfiles, creatorSocials } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { sendWelcomeEmail } from '@/lib/email'
import { sendDiscordNotification } from '@/lib/discord'

export async function POST(req: NextRequest) {
    try {
        const session = await auth()
        if (!session || !session.user) {
            return new NextResponse('Unauthorized', { status: 401 })
        }

        const userId = session.user.id

        const body = await req.json()
        const { name, fullName, avatar, socials, verification, discordHandle } = body

        // 1. Update User Profile
        if (name || avatar || discordHandle) {
            await db
                .update(users)
                .set({
                    ...(name && { name }),
                    ...(avatar && { image: avatar }),
                    ...(discordHandle && { discordHandle }),
                    updatedAt: new Date()
                })
                .where(eq(users.id, userId))
        }

        // 2. Update/Create Creator Profile Socials
        if (socials || fullName) {
            await db
                .insert(creatorProfiles)
                .values({
                    creatorId: userId,
                    legalName: fullName,
                    instagram: socials?.instagram,
                    tiktok: socials?.tiktok,
                    youtube: socials?.youtube,
                    updatedAt: new Date()
                })
                .onConflictDoUpdate({
                    target: creatorProfiles.creatorId,
                    set: {
                        legalName: fullName,
                        instagram: socials?.instagram,
                        tiktok: socials?.tiktok,
                        youtube: socials?.youtube,
                        updatedAt: new Date()
                    }
                })
        }

        // 3. Save Verified Socials to creator_socials table
        if (verification && verification.verifiedSocials) {
            for (const social of verification.verifiedSocials) {
                await db
                    .insert(creatorSocials)
                    .values({
                        creatorId: userId,
                        platform: social.platform,
                        handle: social.handle,
                        followers: social.followers,
                        verificationMethod: 'manual_scrape',
                        verificationStatus: 'pending', // Still pending admin approval
                        updatedAt: new Date()
                    })
                    .onConflictDoUpdate({
                        target: [creatorSocials.creatorId, creatorSocials.platform],
                        set: {
                            handle: social.handle,
                            followers: social.followers,
                            verificationMethod: 'manual_scrape',
                            updatedAt: new Date()
                        }
                    })
            }
        }

        // 4. Complete Onboarding (but keep isApproved = 0)
        if (verification) {
            await db
                .update(users)
                .set({
                    onboardingComplete: 1,
                    isApproved: 0, // Explicitly ensure pending status
                    updatedAt: new Date()
                })
                .where(eq(users.id, userId))

            // Send welcome email
            if (session.user.email) {
                try {
                    const freshUser = await db.query.users.findFirst({
                        where: eq(users.id, userId)
                    })
                    await sendWelcomeEmail(session.user.email, freshUser?.name || name || 'Creator')
                } catch (emailError) {
                    console.error('Failed to send welcome email:', emailError)
                }
            }

            // Send Discord Notification (direct webhook)
            const { discord } = await import('@/lib/discord-logger')
            await discord.creator(
                'Nouveau Créateur (En attente)',
                `Le créateur **${name}** a terminé l'onboarding et attend la validation.`,
                [
                    { name: 'User ID', value: userId, inline: true },
                    { name: 'Email', value: session.user.email || 'N/A', inline: true },
                    { name: 'Max Followers', value: verification.followers?.toString() || 'N/A', inline: true },
                    { name: 'Socials', value: verification.verifiedSocials?.map((s: any) => `${s.platform}: @${s.handle}`).join('\n') || 'None' }
                ]
            )
        }

        return NextResponse.json({ success: true })
    } catch (error) {
        console.error('Error in onboarding:', error)
        return new NextResponse('Internal Server Error', { status: 500 })
    }
}
