import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { userSettings, creatorProfiles, users } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { auth } from '@/auth'
import { userSettingsSchema } from '@/lib/schemas'
import logger from '@/lib/logger'

// GET - Fetch user settings
export async function GET(request: NextRequest) {
  try {
    const session = await auth()

    if (!session || !session.user) {
      return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
    }

    const userId = session.user.id

    // Get user settings
    let settings = await db.query.userSettings.findFirst({
      where: eq(userSettings.userId, userId)
    })

    // Get creator profile (for payment info)
    const profile = await db.query.creatorProfiles.findFirst({
      where: eq(creatorProfiles.creatorId, userId)
    })

    const user = await db.query.users.findFirst({
      where: eq(users.id, userId)
    })

    // If no settings exist, create default ones
    if (!settings) {
      const newSettings = await db.insert(userSettings).values({
        userId: userId,
        emailNotifications: 1,
        newCampaignsNotifications: 1,
        submissionStatusNotifications: 1,
        paymentNotifications: 1,
        marketingEmails: 0,
        profileVisible: 1,
        statsVisible: 0,
      }).returning()

      settings = newSettings[0]
    }

    return NextResponse.json({
      general: {
        discordHandle: user?.discordHandle || '',
        locale: user?.preferredLocale || 'fr'
      },
      notifications: {
        email: settings.emailNotifications === 1,
        newCampaigns: settings.newCampaignsNotifications === 1,
        submissionStatus: settings.submissionStatusNotifications === 1,
        payments: settings.paymentNotifications === 1,
        marketing: settings.marketingEmails === 1
      },
      privacy: {
        profileVisible: settings.profileVisible === 1,
        statsVisible: settings.statsVisible === 1
      },
      payout: {
        bankName: profile?.bankName || '',
        paypalEmail: profile?.paypalEmail || '',
        currency: user?.currency || 'EUR',
        iban: user?.iban || profile?.iban || '',
        bic: user?.bic || profile?.bic || '',
        accountNumber: user?.accountNumber || '',
        routingNumber: user?.routingNumber || '',
        bankAddress: user?.bankAddress || '',
      },
      billing: {
        firstName: profile?.firstName || '',
        lastName: profile?.lastName || '',
        address: profile?.address || '',
        city: profile?.city || '',
        zip: profile?.zip || '',
        country: profile?.country || ''
      }
    })
  } catch (error) {
    logger.error({ err: error }, 'Get settings error')
    return NextResponse.json(
      { error: 'Erreur lors de la récupération des paramètres' },
      { status: 500 }
    )
  }
}

// PUT - Update user settings
export async function PUT(request: NextRequest) {
  try {
    const session = await auth()

    if (!session || !session.user) {
      return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
    }

    const userId = session.user.id

    const data = await request.json()
    const validation = userSettingsSchema.safeParse(data)

    if (!validation.success) {
      return NextResponse.json({ error: validation.error.errors[0].message }, { status: 400 })
    }

    const { notifications, privacy, payout, general, billing } = validation.data

    if (general) {
      await db.update(users)
        .set({
          discordHandle: general.discordHandle,
          preferredLocale: general.locale
        })
        .where(eq(users.id, userId))
    }

    // Update User Settings
    const existingSettings = await db.query.userSettings.findFirst({
      where: eq(userSettings.userId, userId)
    })

    if (existingSettings) {
      await db.update(userSettings)
        .set({
          emailNotifications: notifications?.email ? 1 : 0,
          newCampaignsNotifications: notifications?.newCampaigns ? 1 : 0,
          submissionStatusNotifications: notifications?.submissionStatus ? 1 : 0,
          paymentNotifications: notifications?.payments ? 1 : 0,
          marketingEmails: notifications?.marketing ? 1 : 0,
          profileVisible: privacy?.profileVisible ? 1 : 0,
          statsVisible: privacy?.statsVisible ? 1 : 0,
          updatedAt: new Date()
        })
        .where(eq(userSettings.userId, userId))
    } else {
      await db.insert(userSettings).values({
        userId: userId,
        emailNotifications: notifications?.email ? 1 : 0,
        newCampaignsNotifications: notifications?.newCampaigns ? 1 : 0,
        submissionStatusNotifications: notifications?.submissionStatus ? 1 : 0,
        paymentNotifications: notifications?.payments ? 1 : 0,
        marketingEmails: notifications?.marketing ? 1 : 0,
        profileVisible: privacy?.profileVisible ? 1 : 0,
        statsVisible: privacy?.statsVisible ? 1 : 0,
      })
    }

    // Update Creator Profile (Payment Info)
    const existingProfile = await db.query.creatorProfiles.findFirst({
      where: eq(creatorProfiles.creatorId, userId)
    })

    if (existingProfile) {
      await db.update(users)
        .set({
          currency: payout?.currency || null,
          iban: payout?.iban || null,
          bic: payout?.bic || null,
          accountNumber: payout?.accountNumber || null,
          routingNumber: payout?.routingNumber || null,
          bankName: payout?.bankName || null,
          bankAddress: payout?.bankAddress || null,
          onboardingComplete: 1
        })
        .where(eq(users.id, session.user.id))

      await db.update(creatorProfiles)
        .set({
          bankName: payout?.bankName || null,
          iban: payout?.iban || null,
          bic: payout?.bic || null,
          paypalEmail: payout?.paypalEmail || null,
          // Billing
          firstName: billing?.firstName || null,
          lastName: billing?.lastName || null,
          legalName: `${billing?.firstName || ''} ${billing?.lastName || ''}`.trim() || null, // Keep legacy sync
          address: billing?.address || null,
          city: billing?.city || null,
          zip: billing?.zip || null,
          country: billing?.country || null,
          updatedAt: new Date()
        })
        .where(eq(creatorProfiles.creatorId, userId))
    } else {
      // If profile doesn't exist (shouldn't happen for creators, but safe to handle)
      await db.insert(creatorProfiles).values({
        creatorId: userId,
        bankName: payout?.bankName || null,
        iban: payout?.iban || null,
        bic: payout?.bic || null,
        paypalEmail: payout?.paypalEmail || null,
        // Billing
        firstName: billing?.firstName || null,
        lastName: billing?.lastName || null,
        legalName: `${billing?.firstName || ''} ${billing?.lastName || ''}`.trim() || null,
        address: billing?.address || null,
        city: billing?.city || null,
        zip: billing?.zip || null,
        country: billing?.country || null,
      })
    }

    // Discord Log
    const { sendDiscordNotification } = await import('@/lib/discord')
    await sendDiscordNotification(
      '⚙️ Paramètres Mis à jour',
      `Le créateur **${session.user.name || session.user.email}** a mis à jour ses paramètres.`,
      'info',
      [{ name: 'User ID', value: userId, inline: true }]
    )

    return NextResponse.json({
      message: 'Paramètres mis à jour avec succès'
    })
  } catch (error) {
    logger.error({ err: error }, 'Update settings error')
    return NextResponse.json(
      { error: 'Erreur lors de la mise à jour des paramètres' },
      { status: 500 }
    )
  }
}