import { NextRequest, NextResponse } from 'next/server'

const TIKTOK_CLIENT_ID = process.env.TIKTOK_CLIENT_ID
const TIKTOK_REDIRECT_URI = process.env.NEXT_PUBLIC_URL + '/api/auth/social/tiktok'

const YOUTUBE_CLIENT_ID = process.env.YOUTUBE_CLIENT_ID
const YOUTUBE_REDIRECT_URI = process.env.NEXT_PUBLIC_URL + '/api/auth/social/youtube'

const INSTAGRAM_CLIENT_ID = process.env.INSTAGRAM_CLIENT_ID
const INSTAGRAM_REDIRECT_URI = process.env.NEXT_PUBLIC_URL + '/api/auth/social/instagram'

export async function POST(req: NextRequest) {
    try {
        const { platform, userId } = await req.json()

        if (!platform || !userId) {
            return NextResponse.json(
                { error: 'Platform and userId are required' },
                { status: 400 }
            )
        }

        let authUrl: string

        switch (platform) {
            case 'tiktok':
                if (!TIKTOK_CLIENT_ID) {
                    return NextResponse.json(
                        { error: 'TikTok OAuth not configured' },
                        { status: 500 }
                    )
                }
                authUrl = `https://www.tiktok.com/v2/auth/authorize/?client_key=${TIKTOK_CLIENT_ID}&scope=user.info.basic,video.list&response_type=code&redirect_uri=${encodeURIComponent(TIKTOK_REDIRECT_URI)}&state=${userId}`
                break

            case 'youtube':
                if (!YOUTUBE_CLIENT_ID) {
                    return NextResponse.json(
                        { error: 'YouTube OAuth not configured' },
                        { status: 500 }
                    )
                }
                const scopes = encodeURIComponent('https://www.googleapis.com/auth/youtube.readonly')
                authUrl = `https://accounts.google.com/o/oauth2/v2/auth?client_id=${YOUTUBE_CLIENT_ID}&redirect_uri=${encodeURIComponent(YOUTUBE_REDIRECT_URI)}&response_type=code&scope=${scopes}&access_type=offline&state=${userId}`
                break

            case 'instagram':
                if (!INSTAGRAM_CLIENT_ID) {
                    return NextResponse.json(
                        { error: 'Instagram OAuth not configured' },
                        { status: 500 }
                    )
                }
                authUrl = `https://api.instagram.com/oauth/authorize?client_id=${INSTAGRAM_CLIENT_ID}&redirect_uri=${encodeURIComponent(INSTAGRAM_REDIRECT_URI)}&scope=user_profile,user_media&response_type=code&state=${userId}`
                break

            default:
                return NextResponse.json(
                    { error: 'Invalid platform' },
                    { status: 400 }
                )
        }

        return NextResponse.json({ success: true, authUrl })

    } catch (error: any) {
        console.error('Social connect error:', error)
        return NextResponse.json(
            { error: error.message || 'Internal server error' },
            { status: 500 }
        )
    }
}
