import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { getInstagramFollowers, getYouTubeSubscribers, getTikTokFollowers } from '@/lib/social-scraper'

export async function POST(req: NextRequest) {
    const session = await auth()
    if (!session || session.user.role !== 'creator') {
        return new NextResponse('Unauthorized', { status: 401 })
    }

    try {
        const { platform, handle } = await req.json()

        if (!platform || !handle) {
            return new NextResponse('Missing platform or handle', { status: 400 })
        }

        let result = null

        if (platform === 'instagram') {
            result = await getInstagramFollowers(handle)
        } else if (platform === 'youtube') {
            result = await getYouTubeSubscribers(handle)
        } else if (platform === 'tiktok') {
            result = await getTikTokFollowers(handle)
        } else {
            return NextResponse.json({ success: false, error: 'Plateforme non supportée (TikTok, YouTube, Instagram uniquement)' })
        }

        if (result) {
            return NextResponse.json({
                success: true,
                data: {
                    handle,
                    platform,
                    followers: result.followers,
                    avatarUrl: result.avatarUrl
                }
            })
        } else {
            return NextResponse.json({ success: false, error: 'Compte introuvable ou inaccessible' })
        }
    } catch (error) {
        console.error('Error searching social profile:', error)
        return new NextResponse('Internal Server Error', { status: 500 })
    }
}
