import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { db } from '@/lib/db'
import { accounts, creatorSocials, users } from '@/lib/db/schema'
import { eq, and } from 'drizzle-orm'
import { google } from 'googleapis'

export async function POST(req: NextRequest) {
    const session = await auth()
    if (!session?.user?.id) return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })

    const { provider } = await req.json()

    if (provider === 'youtube') {
        try {
            // 1. Get Access Token from Accounts table
            const account = await db.query.accounts.findFirst({
                where: and(
                    eq(accounts.userId, session.user.id),
                    eq(accounts.provider, 'google')
                )
            })

            if (!account?.access_token) {
                return NextResponse.json({ error: 'No YouTube account connected' }, { status: 400 })
            }

            // 2. Fetch YouTube Channel Info
            const oauth2Client = new google.auth.OAuth2(
                process.env.AUTH_GOOGLE_ID,
                process.env.AUTH_GOOGLE_SECRET
            )

            oauth2Client.setCredentials({
                access_token: account.access_token,
                refresh_token: account.refresh_token
            })

            // Force refresh if needed (googleapis handles this if refresh_token is set)
            // But we might want to listen to 'tokens' event to save new tokens, 
            // though for a single request it might be overkill. 
            // Let's just rely on googleapis using the refresh_token.

            const youtube = google.youtube({ version: 'v3', auth: oauth2Client })
            const response = await youtube.channels.list({
                part: ['snippet', 'statistics'],
                mine: true
            })

            const channel = response.data.items?.[0]
            if (!channel) {
                return NextResponse.json({ error: 'No YouTube channel found' }, { status: 404 })
            }

            const handle = channel.snippet?.customUrl || channel.snippet?.title || 'Unknown'
            const followers = parseInt(channel.statistics?.subscriberCount || '0')
            const profileUrl = channel.snippet?.thumbnails?.default?.url || ''

            // 3. Save to CreatorSocials
            await db.insert(creatorSocials).values({
                creatorId: session.user.id,
                platform: 'youtube',
                handle,
                followers,
                profileUrl,
                verificationStatus: 'approved', // OAuth is trusted
                verificationMethod: 'oauth'
            }).onConflictDoUpdate({
                target: [creatorSocials.creatorId, creatorSocials.platform],
                set: {
                    handle,
                    followers,
                    profileUrl,
                    verificationStatus: 'approved',
                    updatedAt: new Date()
                }
            })

            // Update User table with channel ID if needed
            await db.update(users)
                .set({ youtubeChannelId: channel.id })
                .where(eq(users.id, session.user.id))

            // Discord Log
            const { sendDiscordNotification } = await import('@/lib/discord')
            await sendDiscordNotification(
                '🔗 YouTube Connecté',
                `Le créateur **${session.user.name || session.user.email}** a connecté sa chaîne YouTube **${handle}** (${followers} abonnés).`,
                'success',
                [
                    { name: 'User ID', value: session.user.id, inline: true },
                    { name: 'Followers', value: followers.toString(), inline: true }
                ]
            )

            return NextResponse.json({ success: true, handle, platform: 'youtube' })

        } catch (error) {
            console.error('YouTube Sync Error:', error)
            return NextResponse.json({ error: 'Failed to sync YouTube channel' }, { status: 500 })
        }
    }

    return NextResponse.json({ error: 'Provider not supported' }, { status: 400 })
}
