import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { creatorProfiles } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { authenticateRequest, requireRole } from '@/lib/auth-middleware'
import { getCreatorStats } from '@/lib/tracking/subscriber-tracker'
import { handleApiError, createApiResponse } from '@/lib/api-response'

export async function GET(request: NextRequest) {
  try {
    const authResult = await authenticateRequest(request)
    if (!authResult.success) {
      return NextResponse.json({ error: authResult.error }, { status: 401 })
    }

    const roleCheck = requireRole(authResult.role, 'creator')
    if (!roleCheck.allowed) {
      return NextResponse.json({ error: roleCheck.error }, { status: 403 })
    }

    // Récupérer le profil du créateur
    const profile = await db.query.creatorProfiles.findFirst({
      where: eq(creatorProfiles.creatorId, authResult.userId!)
    })

    if (!profile) {
      return NextResponse.json(
        { error: 'Profil créateur non trouvé' },
        { status: 404 }
      )
    }

    // Construire la config des plateformes
    const platforms: Record<string, any> = {}

    if (profile.instagram) {
      platforms.instagram = { username: profile.instagram }
    }
    if (profile.youtube) {
      platforms.youtube = { channelId: profile.youtube }
    }
    if (profile.tiktok) {
      platforms.tiktok = { username: profile.tiktok }
    }

    if (Object.keys(platforms).length === 0) {
      return NextResponse.json(
        { error: 'Aucune plateforme configurée' },
        { status: 400 }
      )
    }

    // Récupérer les stats en temps réel
    const stats = await getCreatorStats(authResult.userId!, platforms)

    // Calculer le total des abonnés
    let totalFollowers = 0
    let successfulPlatforms = 0

    for (const platformStats of Object.values(stats) as any[]) {
      if (platformStats.success) {
        totalFollowers += platformStats.followers
        successfulPlatforms++
      }
    }

    return createApiResponse({
      platforms: stats,
      totalFollowers,
      successfulPlatforms,
      totalPlatforms: Object.keys(platforms).length,
      lastUpdated: new Date().toISOString()
    })

  } catch (error) {
    return handleApiError(error)
  }
}