import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { handleApiError, createApiResponse } from '@/lib/api-response'
import { db } from '@/lib/db'
import { users } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { saveFile } from '@/lib/storage'

const MAX_FILE_SIZE = 5 * 1024 * 1024 // 5MB
const ALLOWED_TYPES = ['image/jpeg', 'image/png', 'image/gif', 'image/webp']

export async function POST(request: NextRequest) {
  try {
    const session = await auth()
    if (!session || !session.user) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    if ((session.user as any).role !== 'creator') {
      return NextResponse.json({ error: 'Forbidden' }, { status: 403 })
    }

    const userId = session.user.id

    let formData;
    try {
      formData = await request.formData()
    } catch (e) {
      return NextResponse.json({ error: 'Invalid form data' }, { status: 400 })
    }

    const file = formData.get('avatar') as File

    if (!file) {
      return NextResponse.json({ error: 'Fichier avatar requis' }, { status: 400 })
    }

    if (file.size > MAX_FILE_SIZE) {
      return NextResponse.json({ error: 'Fichier trop volumineux (max 5MB)' }, { status: 400 })
    }

    if (!ALLOWED_TYPES.includes(file.type)) {
      return NextResponse.json({ error: 'Type de fichier non autorisé' }, { status: 400 })
    }

    // Save using S3 Storage Lib
    const result = await saveFile(file, 'avatars')

    if (result.error) {
      return NextResponse.json({ error: result.error }, { status: 500 })
    }

    const publicUrl = result.path
    console.log('[UploadAPI] Generated publicUrl:', publicUrl)

    // Mettre à jour l'avatar dans la base de données
    await db.update(users)
      .set({
        image: publicUrl,
        updatedAt: new Date()
      })
      .where(eq(users.id, userId))

    return createApiResponse({
      avatarUrl: publicUrl,
      fileName: publicUrl.split('/').pop(),
      size: result.size,
      type: file.type,
      uploadedAt: new Date().toISOString()
    })

  } catch (error) {
    console.error(error)
    return handleApiError(error)
  }
}