import { NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { submissions, campaigns, users } from '@/lib/db/schema'
import { eq, and, lt } from 'drizzle-orm'
import { sendDiscordNotification } from '@/lib/discord'

export async function GET(request: Request) {
    try {
        // Check for Authorization header (optional, for security if triggered externally)
        // const authHeader = request.headers.get('authorization')
        // if (authHeader !== `Bearer ${process.env.CRON_SECRET}`) {
        //   return new NextResponse('Unauthorized', { status: 401 })
        // }

        // Calculate date 5 days ago
        const fiveDaysAgo = new Date()
        fiveDaysAgo.setDate(fiveDaysAgo.getDate() - 5)

        // Find pending submissions older than 5 days
        const pendingSubmissions = await db.query.submissions.findMany({
            where: and(
                eq(submissions.status, 'pending'),
                lt(submissions.submittedAt, fiveDaysAgo)
            ),
            with: {
                campaign: true,
                creator: true
            },
            limit: 50 // Process in batches to avoid OOM
        })

        if (pendingSubmissions.length === 0) {
            return NextResponse.json({ message: 'No submissions to auto-validate' })
        }

        const results = []

        for (const submission of pendingSubmissions) {
            // Update status to approved
            await db.update(submissions)
                .set({
                    status: 'approved',
                    reviewedAt: new Date()
                })
                .where(eq(submissions.id, submission.id))

            // Log to Discord
            await sendDiscordNotification(
                '⚖️ Auto-Validation (CGU Art. 6)',
                `Submission auto-validated after 5 days of inactivity.`,
                'info',
                [
                    { name: 'Campaign', value: submission.campaign.title, inline: true },
                    { name: 'Creator', value: submission.creator.name || submission.creator.email, inline: true },
                    { name: 'Submission ID', value: submission.id, inline: false }
                ]
            )

            results.push(submission.id)
        }

        return NextResponse.json({
            success: true,
            count: results.length,
            validatedIds: results
        })

    } catch (error) {
        console.error('Auto-validation cron error:', error)
        await sendDiscordNotification(
            '🚨 Auto-Validation Error',
            'Failed to execute auto-validation cron.',
            'error'
        )
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}
