import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { submissions, submissionStatsHistory } from '@/lib/db/schema'
import { eq, and, or, ne } from 'drizzle-orm'
import { SocialScraper } from '@/lib/social-scraper'

export async function GET(req: NextRequest) {
    // Simple security check (Cron secret)
    const authHeader = req.headers.get('authorization')
    if (process.env.CRON_SECRET && authHeader !== `Bearer ${process.env.CRON_SECRET}`) {
        return new NextResponse('Unauthorized', { status: 401 })
    }

    try {
        // Fetch active submissions (approved or pending)
        // We don't need to update rejected ones
        const activeSubmissions = await db.query.submissions.findMany({
            where: or(
                eq(submissions.status, 'approved'),
                eq(submissions.status, 'pending')
            ),
            limit: 20, // Process in batches to avoid timeouts
            orderBy: (submissions, { asc }) => [asc(submissions.lastUpdated)] // Update oldest first
        })

        if (activeSubmissions.length === 0) {
            return NextResponse.json({ processed: 0 })
        }

        let processed = 0
        let updated = 0

        for (const submission of activeSubmissions) {
            processed++
            const scraper = new SocialScraper()
            const stats = await scraper.getStats(submission.platform, submission.videoUrl)

            if (stats) {
                await db.update(submissions)
                    .set({
                        viewsCount: stats.views > 0 ? stats.views : submission.viewsCount, // Only update if we got valid views
                        likes: stats.likes,
                        engagementRate: stats.engagementRate.toString(),
                        lastUpdated: new Date()
                    })
                    .where(eq(submissions.id, submission.id))

                // Log history
                await db.insert(submissionStatsHistory).values({
                    submissionId: submission.id,
                    views: stats.views,
                    likes: stats.likes,
                    engagementRate: stats.engagementRate.toString(),
                    recordedAt: new Date()
                })

                updated++
            } else {
                // Just update timestamp so we don't retry immediately
                await db.update(submissions)
                    .set({ lastUpdated: new Date() })
                    .where(eq(submissions.id, submission.id))
            }
        }

        return NextResponse.json({ processed, updated })
    } catch (error) {
        console.error('Error in update-stats cron:', error)
        return new NextResponse('Internal Error', { status: 500 })
    }
}
