import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { submissions, campaigns, users } from '@/lib/db/schema'
import { eq, and, gte, lte, sql } from 'drizzle-orm'
import { requireAdmin } from '@/lib/auth-middleware'

export async function GET(request: NextRequest) {
  const authResult = await requireAdmin(request)
  if (authResult.response) return authResult.response

  try {
    const { searchParams } = new URL(request.url)
    const period = searchParams.get('period') || '30d'

    // Calculate date range
    const now = new Date()
    const startDate = new Date()
    
    switch (period) {
      case '7d':
        startDate.setDate(now.getDate() - 7)
        break
      case '30d':
        startDate.setDate(now.getDate() - 30)
        break
      case '90d':
        startDate.setDate(now.getDate() - 90)
        break
      case '1y':
        startDate.setFullYear(now.getFullYear() - 1)
        break
      default:
        startDate.setDate(now.getDate() - 30)
    }

    // Get total revenue from approved submissions
    const revenueData = await db
      .select({
        totalRevenue: sql<number>`COALESCE(SUM(CAST(${submissions.revenue} AS DECIMAL)), 0)`,
        totalSubmissions: sql<number>`COUNT(*)`,
        approvedSubmissions: sql<number>`COUNT(CASE WHEN ${submissions.status} = 'approved' THEN 1 END)`,
      })
      .from(submissions)
      .where(
        and(
          gte(submissions.submittedAt, startDate),
          lte(submissions.submittedAt, now)
        )
      )

    // Get revenue by campaign
    const revenueByCampaign = await db
      .select({
        campaignId: campaigns.id,
        campaignTitle: campaigns.title,
        studioName: users.company,
        revenue: sql<number>`COALESCE(SUM(CAST(${submissions.revenue} AS DECIMAL)), 0)`,
        submissions: sql<number>`COUNT(${submissions.id})`,
      })
      .from(submissions)
      .leftJoin(campaigns, eq(submissions.campaignId, campaigns.id))
      .leftJoin(users, eq(campaigns.studioId, users.id))
      .where(
        and(
          gte(submissions.submittedAt, startDate),
          lte(submissions.submittedAt, now),
          eq(submissions.status, 'approved')
        )
      )
      .groupBy(campaigns.id, campaigns.title, users.company)
      .limit(10)

    // Get revenue by creator
    const revenueByCreator = await db
      .select({
        creatorId: users.id,
        creatorName: users.name,
        creatorEmail: users.email,
        revenue: sql<number>`COALESCE(SUM(CAST(${submissions.revenue} AS DECIMAL)), 0)`,
        submissions: sql<number>`COUNT(${submissions.id})`,
      })
      .from(submissions)
      .leftJoin(users, eq(submissions.creatorId, users.id))
      .where(
        and(
          gte(submissions.submittedAt, startDate),
          lte(submissions.submittedAt, now),
          eq(submissions.status, 'approved')
        )
      )
      .groupBy(users.id, users.name, users.email)
      .orderBy(sql`SUM(CAST(${submissions.revenue} AS DECIMAL)) DESC`)
      .limit(10)

    // Calculate platform fee (10% of total revenue)
    const platformFee = (revenueData[0]?.totalRevenue || 0) * 0.10
    const creatorRevenue = (revenueData[0]?.totalRevenue || 0) * 0.90

    return NextResponse.json({
      period,
      totalRevenue: revenueData[0]?.totalRevenue || 0,
      platformRevenue: platformFee,
      creatorRevenue: creatorRevenue,
      totalSubmissions: revenueData[0]?.totalSubmissions || 0,
      approvedSubmissions: revenueData[0]?.approvedSubmissions || 0,
      revenueByCampaign: revenueByCampaign,
      revenueByCreator: revenueByCreator,
    })
  } catch (error) {
    console.error('Error fetching revenue analytics:', error)
    return NextResponse.json(
      { error: 'Failed to fetch revenue analytics' },
      { status: 500 }
    )
  }
}
