import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { db } from '@/lib/db'
import { campaigns } from '@/lib/db/schema'
import { sql } from 'drizzle-orm'

export async function GET(request: NextRequest) {
  try {
    const session = await auth()
    if (!session || session.user.role !== 'admin') {
      return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
    }

    const campaignsData = await db.query.campaigns.findMany({
      with: {
        studio: true,
        submissions: true
      },
      orderBy: (campaigns, { desc }) => [desc(campaigns.createdAt)]
    })

    const campaignsWithStats = campaignsData.map(campaign => ({
      id: campaign.id,
      title: campaign.title,
      description: campaign.description,
      budget: parseFloat(campaign.budget || '0'),
      status: campaign.status,
      studioName: campaign.studio?.name || campaign.studio?.email || 'Studio inconnu',
      submissions: campaign.submissions?.length || 0,
      views: campaign.submissions?.reduce((sum, sub) => sum + (sub.views || 0), 0) || 0,
      createdAt: campaign.createdAt
    }))

    return NextResponse.json({ campaigns: campaignsWithStats })
  } catch (error) {
    console.error('Error fetching campaigns:', error)
    return NextResponse.json({ error: 'Erreur serveur' }, { status: 500 })
  }
}
