import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { invoices, campaigns, users } from '@/lib/db/schema'
import { authenticateRequest, requireRole } from '@/lib/auth-middleware'
import { desc, sql, eq } from 'drizzle-orm'

export async function GET(request: NextRequest) {
    try {
        const authResult = await authenticateRequest(request)
        if (!authResult.success) return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })

        const roleCheck = requireRole(authResult.role, 'admin')
        if (!roleCheck.allowed) return NextResponse.json({ error: 'Forbidden' }, { status: 403 })

        // Calculate totals
        const revenueResult = await db.select({
            total: sql<number>`sum(${invoices.amount})`,
        }).from(invoices).where(eq(invoices.status, 'paid'))

        const activeCampaigns = await db.select({ count: sql<number>`count(*)` }).from(campaigns).where(eq(campaigns.status, 'active'))
        const activeUsers = await db.select({ count: sql<number>`count(*)` }).from(users).where(eq(users.role, 'creator')) // Simplified

        const monthlyData = await db.select({
            month: sql<string>`to_char(${invoices.createdAt}, 'Mon')`,
            amount: sql<number>`sum(${invoices.amount})`
        })
            .from(invoices)
            .where(eq(invoices.status, 'paid'))
            .groupBy(sql`to_char(${invoices.createdAt}, 'Mon')`, sql`date_trunc('month', ${invoices.createdAt})`)
            .orderBy(sql`date_trunc('month', ${invoices.createdAt})`)

        // Calculate current month revenue
        const currentMonth = new Date().toLocaleString('en-US', { month: 'short' })
        const currentMonthData = monthlyData.find(d => d.month === currentMonth)
        const monthlyRevenue = currentMonthData?.amount || 0

        // Calculate growth (simple prev month comparison)
        const prevMonthDate = new Date(); prevMonthDate.setMonth(prevMonthDate.getMonth() - 1);
        const prevMonth = prevMonthDate.toLocaleString('en-US', { month: 'short' })
        const prevMonthData = monthlyData.find(d => d.month === prevMonth)
        const prevAmount = prevMonthData?.amount || 0
        const monthlyGrowth = prevAmount > 0 ? ((monthlyRevenue - prevAmount) / prevAmount) * 100 : 0

        // Top Studios
        const topStudios = await db.select({
            name: users.name,
            totalSpent: sql<number>`sum(${invoices.amount})`
        })
            .from(invoices)
            .leftJoin(users, eq(invoices.studioId, users.id))
            .where(eq(invoices.status, 'paid'))
            .groupBy(users.name)
            .orderBy(sql`sum(${invoices.amount}) desc`)
            .limit(5)

        return NextResponse.json({
            totalRevenue: revenueResult[0]?.total || 0,
            monthlyRevenue,
            monthlyGrowth,
            activeCampaigns: activeCampaigns[0]?.count || 0, // Should filter by date range properly but ok for snapshot
            activeUsers: activeUsers[0]?.count || 0,
            monthlyData,
            topStudios,
            maxMonthly: Math.max(...monthlyData.map(d => Number(d.amount)), 1)
        })
    } catch (e) {
        console.error(e)
        return NextResponse.json({ error: 'Internal Error' }, { status: 500 })
    }
}
