import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { users } from '@/lib/db/schema'
import { authenticateRequest, requireRole } from '@/lib/auth-middleware'
import { desc, eq, and } from 'drizzle-orm'

export async function GET(request: NextRequest) {
    try {
        const authResult = await authenticateRequest(request)
        if (!authResult.success) return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })

        const roleCheck = requireRole(authResult.role, 'admin')
        if (!roleCheck.allowed) return NextResponse.json({ error: 'Forbidden' }, { status: 403 })

        const data = await db.query.users.findMany({
            where: and(
                eq(users.role, 'creator'),
                eq(users.isApproved, 0)
            ),
            orderBy: [desc(users.createdAt)]
        })

        // Map to expected format
        const formatted = data.map(u => ({
            id: u.id,
            type: 'creator_approval',
            status: 'pending',
            createdAt: u.createdAt,
            user: u,
        }))

        return NextResponse.json(formatted)
    } catch (e) {
        console.error(e)
        return NextResponse.json({ error: 'Internal Error' }, { status: 500 })
    }
}
