import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { campaignAssets, campaigns } from '@/lib/db/schema'
import { saveFile } from '@/lib/storage'
import { eq } from 'drizzle-orm'

export async function POST(req: NextRequest) {
    try {
        const formData = await req.formData()
        const file = formData.get('file') as File
        const campaignId = formData.get('campaignId') as string
        const fileType = formData.get('fileType') as string || 'other'

        if (!file || !campaignId) {
            return NextResponse.json(
                { error: 'File and campaignId are required' },
                { status: 400 }
            )
        }

        // Get campaign
        const campaign = await db.query.campaigns.findFirst({
            where: eq(campaigns.id, campaignId)
        })

        if (!campaign) {
            return NextResponse.json(
                { error: 'Campaign not found' },
                { status: 404 }
            )
        }

        // Upload to S3/MinIO (using campaignId as subdir for organization)
        const uploadResult = await saveFile(file, campaignId)

        if (uploadResult.error) {
            return NextResponse.json(
                { error: uploadResult.error || 'Upload failed' },
                { status: 500 }
            )
        }

        // Save metadata to database
        const [asset] = await db.insert(campaignAssets).values({
            campaignId,
            fileId: uploadResult.path, // Using path/url as ID
            fileName: file.name,
            originalName: file.name,
            mimeType: file.type,
            fileType,
            driveUrl: uploadResult.path, // Compatibility
            size: uploadResult.size,
        }).returning()

        return NextResponse.json({
            success: true,
            asset: {
                id: asset.id,
                fileName: asset.fileName,
                fileType: asset.fileType,
                driveUrl: asset.driveUrl,
                size: asset.size,
                uploadedAt: asset.uploadedAt,
            }
        })

    } catch (error: any) {
        console.error('Asset upload error:', error)
        return NextResponse.json(
            { error: error.message || 'Internal server error' },
            { status: 500 }
        )
    }
}
