import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { invoices, campaigns, users } from '@/lib/db/schema'
import { eq, and } from 'drizzle-orm'
import { authenticateRequest } from '@/lib/auth-middleware'
import { generateMandateInvoice } from '@/lib/pdf/generator'
import { z } from 'zod'

const generateSchema = z.object({
    campaignId: z.string().uuid(),
})

export async function POST(request: NextRequest) {
    try {
        const authResult = await authenticateRequest(request)
        if (!authResult.success) {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
        }

        const { userId, role } = authResult
        if (role !== 'studio') {
            return NextResponse.json({ error: 'Forbidden' }, { status: 403 })
        }

        const body = await request.json()
        const validation = generateSchema.safeParse(body)
        if (!validation.success) {
            return NextResponse.json({ error: 'Invalid ID' }, { status: 400 })
        }

        const { campaignId } = validation.data

        // Fetch campaign
        const campaign = await db.query.campaigns.findFirst({
            where: and(
                eq(campaigns.id, campaignId),
                eq(campaigns.studioId, userId!)
            ),
            with: {
                studio: true
            }
        })

        if (!campaign) {
            return NextResponse.json({ error: 'Campaign not found' }, { status: 404 })
        }

        // Check if invoice already exists
        const existingInvoice = await db.query.invoices.findFirst({
            where: eq(invoices.campaignId, campaignId)
        })

        if (existingInvoice) {
            return NextResponse.json({
                success: true,
                invoiceId: existingInvoice.id,
                invoiceNumber: existingInvoice.invoiceNumber,
                status: existingInvoice.status
            })
        }

        // Generate Invoice Number: INV-YYYYMM-{CAMPAIGN_SHORT_ID}
        const date = new Date()
        const invoiceNumber = `INV-${date.getFullYear()}${(date.getMonth() + 1).toString().padStart(2, '0')}-${campaign.id.slice(0, 8).toUpperCase()}`

        // Generate PDF (Stub for now or specific generator)
        // We can reuse generateMandateInvoice logic or link to a dynamic route that generates it on fly.
        // For now, let's create the record.
        const pdfUrl = `/api/invoices/${invoiceNumber}/pdf` // Placeholder URL

        const [newInvoice] = await db.insert(invoices).values({
            studioId: userId!,
            campaignId: campaignId,
            amount: campaign.budget, // Assume full budget is billed
            currency: 'EUR', // Default or from campaign if it had currency
            invoiceNumber: invoiceNumber,
            pdfUrl: pdfUrl,
            status: 'pending',
            createdAt: new Date(),
        }).returning()

        return NextResponse.json({
            success: true,
            invoiceId: newInvoice.id,
            invoiceNumber: newInvoice.invoiceNumber,
            status: newInvoice.status
        })

    } catch (error) {
        console.error('Error generating invoice:', error)
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}
