import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { invoices } from '@/lib/db/schema'
import { eq, and } from 'drizzle-orm'
import { authenticateRequest } from '@/lib/auth-middleware'
import { z } from 'zod'
import { sendDiscordNotification } from '@/lib/discord'

const declareSchema = z.object({
    invoiceId: z.string().uuid(),
})

export async function POST(request: NextRequest) {
    try {
        const authResult = await authenticateRequest(request)
        if (!authResult.success) {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
        }

        const { userId, role } = authResult

        if (role !== 'studio') {
            return NextResponse.json({ error: 'Forbidden' }, { status: 403 })
        }

        const body = await request.json()
        const validation = declareSchema.safeParse(body)

        if (!validation.success) {
            return NextResponse.json({ error: 'Invalid request' }, { status: 400 })
        }

        const { invoiceId } = validation.data

        // Find invoice and verify ownership
        const invoice = await db.query.invoices.findFirst({
            where: and(
                eq(invoices.id, invoiceId),
                eq(invoices.studioId, userId!)
            )
        })

        if (!invoice) {
            return NextResponse.json({ error: 'Invoice not found' }, { status: 404 })
        }

        if (invoice.status === 'paid') {
            return NextResponse.json({ error: 'Invoice already paid' }, { status: 400 })
        }

        // Update status
        await db.update(invoices)
            .set({ status: 'pending_verification' })
            .where(eq(invoices.id, invoiceId))

        // Notify Admin (Discord)
        await sendDiscordNotification(
            '💸 Virement Déclaré',
            `Le Studio **${userId}** a déclaré avoir effectué le virement pour la facture **${invoice.invoiceNumber}** (${invoice.amount} ${invoice.currency || 'EUR'}).`,
            'warning'
        )

        return NextResponse.json({ success: true, status: 'pending_verification' })
    } catch (error) {
        console.error('Error declaring payment:', error)
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}
