import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { invoices } from '@/lib/db/schema'
import { eq, desc } from 'drizzle-orm'
import { authenticateRequest } from '@/lib/auth-middleware'

export async function GET(request: NextRequest) {
    try {
        const authResult = await authenticateRequest(request)
        if (!authResult.success) {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
        }

        const { userId, role } = authResult

        if (role !== 'studio') {
            return NextResponse.json({ error: 'Forbidden' }, { status: 403 })
        }

        const studioInvoices = await db.query.invoices.findMany({
            where: eq(invoices.studioId, userId!),
            orderBy: [desc(invoices.createdAt)],
            columns: {
                id: true,
                amount: true,
                currency: true,
                status: true,
                createdAt: true,
                pdfUrl: true,
                invoiceNumber: true,
            }
        })

        // Map status to ensure frontend compatibility
        // PENDING -> Attente virement
        // PENDING_VERIFICATION -> Vérification en cours
        // PAID -> Payé

        return NextResponse.json(studioInvoices)
    } catch (error) {
        console.error('Error fetching studio payments:', error)
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}
