import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { users } from '@/lib/db/schema'
import { sendAccessRequestNotification } from '@/lib/email'
import { eq } from 'drizzle-orm'

export async function POST(request: NextRequest) {
  try {
    const { email, company, name } = await request.json()

    if (!email || !company || !name) {
      return NextResponse.json(
        { error: 'Email, entreprise et nom requis' },
        { status: 400 }
      )
    }

    // Check if already exists
    const existingUser = await db.query.users.findFirst({
      where: (users, { eq }) => eq(users.email, email)
    })

    if (existingUser) {
      // Update existing user with name and company
      await db.update(users)
        .set({
          name,
          company,
          updatedAt: new Date()
        })
        .where(eq(users.id, existingUser.id))
    } else {
      // Create pending studio account
      await db.insert(users).values({
        email,
        company,
        name,
        role: 'studio',
        isApproved: 0, // Pending approval
      })
    }

    // Send notification to admin
    await sendAccessRequestNotification(email, company)

    // Discord log
    const { discord } = await import('@/lib/discord-logger')
    await discord.studio(
      'Nouvelle Demande Studio',
      `**${name}** de **${company}** demande un accès studio.`,
      [
        { name: 'Email', value: email, inline: true },
        { name: 'Entreprise', value: company, inline: true }
      ]
    )

    return NextResponse.json({
      message: 'Demande envoyée avec succès',
    })
  } catch (error) {
    console.error('Studio access request error:', error)
    return NextResponse.json(
      { error: 'Erreur lors de la demande d\'accès' },
      { status: 500 }
    )
  }
}
