import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { campaigns, submissions, campaignApplications, users } from '@/lib/db/schema'
import { eq, and, sum, count, desc, sql } from 'drizzle-orm'
import { authenticateRequest, requireRole } from '@/lib/auth-middleware'

export async function GET(request: NextRequest) {
    try {
        const authResult = await authenticateRequest(request)
        if (!authResult.success) {
            return NextResponse.json({ error: authResult.error }, { status: 401 })
        }

        const roleCheck = requireRole(authResult.role, 'studio')
        if (!roleCheck.allowed) {
            return NextResponse.json({ error: roleCheck.error }, { status: 403 })
        }

        const studioId = authResult.userId

        // Get all campaigns for this studio
        const studioCampaigns = await db
            .select()
            .from(campaigns)
            .where(eq(campaigns.studioId, studioId))

        const campaignIds = studioCampaigns.map(c => c.id)

        if (campaignIds.length === 0) {
            return NextResponse.json({
                totalViews: 0,
                totalSpent: 0,
                activeCampaigns: 0,
                totalCampaigns: 0,
                totalCreators: 0,
                totalSubmissions: 0,
                avgCPM: 0,
                topCampaigns: []
            })
        }

        // Get submission stats
        const submissionStats = await db
            .select({
                campaignId: submissions.campaignId,
                totalViews: sum(submissions.viewsCount),
                totalRevenue: sum(submissions.revenue),
                submissionCount: count()
            })
            .from(submissions)
            .where(sql`${submissions.campaignId} IN (${sql.raw(campaignIds.map(id => `'${id}'`).join(','))})`)
            .groupBy(submissions.campaignId)

        // Calculate totals
        const totalViews = submissionStats.reduce((acc, s) => acc + (Number(s.totalViews) || 0), 0)
        const totalSpent = submissionStats.reduce((acc, s) => acc + (Number(s.totalRevenue) || 0), 0)
        const totalSubmissions = submissionStats.reduce((acc, s) => acc + (Number(s.submissionCount) || 0), 0)

        // Active campaigns count
        const activeCampaigns = studioCampaigns.filter(c => c.status === 'active').length

        // Unique creators count
        const creatorResult = await db
            .selectDistinct({ creatorId: campaignApplications.creatorId })
            .from(campaignApplications)
            .where(
                and(
                    sql`${campaignApplications.campaignId} IN (${sql.raw(campaignIds.map(id => `'${id}'`).join(','))})`,
                    eq(campaignApplications.status, 'approved')
                )
            )
        const totalCreators = creatorResult.length

        // Calculate average CPM
        const avgCPM = totalViews > 0 ? (totalSpent / totalViews) * 1000 : 0

        // Top campaigns by views
        const topCampaigns = studioCampaigns
            .map(campaign => {
                const stats = submissionStats.find(s => s.campaignId === campaign.id)
                return {
                    id: campaign.id,
                    title: campaign.title,
                    views: Number(stats?.totalViews) || 0,
                    spent: Number(stats?.totalRevenue) || 0
                }
            })
            .sort((a, b) => b.views - a.views)
            .slice(0, 5)

        return NextResponse.json({
            totalViews,
            totalSpent,
            activeCampaigns,
            totalCampaigns: studioCampaigns.length,
            totalCreators,
            totalSubmissions,
            avgCPM,
            topCampaigns
        })

    } catch (error) {
        console.error('Studio stats error:', error)
        return NextResponse.json({ error: 'Erreur serveur' }, { status: 500 })
    }
}
