import { NextRequest, NextResponse } from 'next/server'
import { authenticateRequest, requireRole } from '@/lib/auth-middleware'
import { saveFile, deleteFile } from '@/lib/storage'
import { handleApiError, createApiResponse } from '@/lib/api-response'
import { db } from '@/lib/db'
import { campaigns, campaignAssets } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'

export async function POST(request: NextRequest) {
  try {
    const authResult = await authenticateRequest(request)
    if (!authResult.success) {
      return NextResponse.json({ error: authResult.error }, { status: 401 })
    }

    const roleCheck = requireRole(authResult.role, 'studio')
    if (!roleCheck.allowed) {
      return NextResponse.json({ error: roleCheck.error }, { status: 403 })
    }

    const formData = await request.formData()
    const file = formData.get('file') as File
    const campaignId = formData.get('campaignId') as string

    if (!file) {
      return NextResponse.json({ error: 'Fichier requis' }, { status: 400 })
    }

    if (!campaignId) {
      return NextResponse.json({ error: 'ID de campagne requis' }, { status: 400 })
    }

    // Check campaign ownership
    const campaign = await db.query.campaigns.findFirst({
      where: eq(campaigns.id, campaignId)
    })

    if (!campaign) {
      return NextResponse.json({ error: 'Campagne non trouvée' }, { status: 404 })
    }

    if (campaign.studioId !== authResult.userId) {
      return NextResponse.json({ error: 'Accès non autorisé à cette campagne' }, { status: 403 })
    }

    // Check file size (5GB limit)
    const MAX_FILE_SIZE = 5 * 1024 * 1024 * 1024
    if (file.size > MAX_FILE_SIZE) {
      return NextResponse.json({ error: 'Fichier trop volumineux (max 5GB)' }, { status: 400 })
    }

    // Check total campaign storage
    const existingSizeResult: any = await db.query.campaign_assets.findMany({
      where: eq(campaignAssets.campaignId, campaignId),
      columns: { size: true }
    })

    const totalExisting = existingSizeResult.reduce((acc: number, r: any) => acc + (Number(r.size) || 0), 0)
    if ((totalExisting + file.size) > MAX_FILE_SIZE) {
      return NextResponse.json({ error: 'Limite de stockage de la campagne atteinte (5GB)' }, { status: 400 })
    }

    // Allowed types
    const allowedTypes = [
      'image/jpeg', 'image/png', 'image/gif', 'image/webp',
      'video/mp4', 'video/webm', 'video/avi',
      'audio/mp3', 'audio/wav', 'audio/mpeg'
    ]

    if (!allowedTypes.includes(file.type)) {
      return NextResponse.json({ error: 'Type de fichier non autorisé' }, { status: 400 })
    }

    // Upload to MinIO/S3
    // Use campaignId as sub-directory
    const uploadResult = await saveFile(file, campaignId)

    if (uploadResult.error) {
      return NextResponse.json({ error: uploadResult.error }, { status: 500 })
    }

    // Save metadata
    try {
      await db.insert(campaignAssets).values({
        campaignId: campaignId,
        fileId: uploadResult.path, // Using path/url as ID/Key for now, or just storing URL
        fileName: file.name, // Keeping original name or generated? saveFile generates unique name but returns path.
        originalName: file.name,
        mimeType: file.type,
        size: String(uploadResult.size),
        driveUrl: uploadResult.path, // Mapping path to driveUrl for compatibility
        uploadedAt: new Date()
      })
    } catch (e: any) {
      console.error('Erreur enregistrement campaign asset:', e)
      // Rollback
      await deleteFile(uploadResult.path)
      return NextResponse.json({ error: 'Erreur lors de l\'enregistrement du fichier' }, { status: 500 })
    }

    return createApiResponse({
      fileId: uploadResult.path,
      url: uploadResult.path,
      fileName: file.name,
      originalName: file.name,
      size: uploadResult.size,
      type: file.type,
      uploadedAt: new Date().toISOString()
    })

  } catch (error) {
    return handleApiError(error)
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const authResult = await authenticateRequest(request)
    if (!authResult.success) {
      return NextResponse.json({ error: authResult.error }, { status: 401 })
    }

    const roleCheck = requireRole(authResult.role, 'studio')
    if (!roleCheck.allowed) {
      return NextResponse.json({ error: roleCheck.error }, { status: 403 })
    }

    const { searchParams } = new URL(request.url)
    const fileUrl = searchParams.get('fileId') // Using fileId param but treating as URL/Key

    if (!fileUrl) {
      return NextResponse.json({ error: 'ID de fichier (URL) requis' }, { status: 400 })
    }

    const deleteResult = await deleteFile(fileUrl)

    if (!deleteResult.success) {
      return NextResponse.json({ error: deleteResult.error || 'Erreur lors de la suppression' }, { status: 500 })
    }

    // Also delete from DB
    await db.delete(campaignAssets).where(eq(campaignAssets.driveUrl, fileUrl))

    return createApiResponse({ message: 'Fichier supprimé avec succès' })

  } catch (error) {
    return handleApiError(error)
  }
}