import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { submissions, campaigns, users } from '@/lib/db/schema'
import { eq, and, desc } from 'drizzle-orm'
import { authenticateRequest, requireRole } from '@/lib/auth-middleware'
import { z } from 'zod'

const createSubmissionSchema = z.object({
  campaignId: z.string().uuid('Invalid campaign ID'),
  videoUrl: z.string().min(1, 'Video URL or path is required'),
  platform: z.enum(['tiktok', 'youtube', 'instagram']),
})

// GET /api/submissions - List submissions
export async function GET(request: NextRequest) {
  try {
    const authResult = await authenticateRequest(request)
    if (!authResult.success) {
      return NextResponse.json({ error: authResult.error }, { status: 401 })
    }

    const { userId, role } = authResult

    let submissionList

    if (role === 'creator') {
      // Creators see only their submissions
      submissionList = await db.query.submissions.findMany({
        where: eq(submissions.creatorId, userId!),
        orderBy: [desc(submissions.submittedAt)],
        with: {
          campaign: true,
        },
      })
    } else if (role === 'studio') {
      // Studios see submissions for their campaigns
      submissionList = await db.query.submissions.findMany({
        where: eq(campaigns.studioId, userId!),
        orderBy: [desc(submissions.submittedAt)],
        with: {
          campaign: true,
          creator: true,
        },
      })
    } else {
      return NextResponse.json({ error: 'Forbidden' }, { status: 403 })
    }

    return NextResponse.json(submissionList)
  } catch (error) {
    console.error('Get submissions error:', error)
    return NextResponse.json(
      { error: 'Failed to fetch submissions' },
      { status: 500 }
    )
  }
}

// POST /api/submissions - Create submission (Creator only)
export async function POST(request: NextRequest) {
  const ip = request.headers.get('x-forwarded-for') || 'unknown'
  const { rateLimiter } = await import('@/lib/rate-limit')

  // 20 submissions per minute
  if (!rateLimiter.check(20, ip)) {
    return NextResponse.json({ error: 'Too many requests' }, { status: 429 })
  }

  try {
    const authResult = await authenticateRequest(request)
    if (!authResult.success) {
      return NextResponse.json({ error: authResult.error }, { status: 401 })
    }

    const roleCheck = requireRole(authResult.role, 'creator')
    if (!roleCheck.allowed) {
      return NextResponse.json({ error: roleCheck.error }, { status: 403 })
    }

    // Check if creator is approved
    const user = await db.query.users.findFirst({
      where: eq(users.id, authResult.userId!)
    })

    if (!user || user.isApproved !== 1) {
      return NextResponse.json({ error: 'Account not approved' }, { status: 403 })
    }

    const body = await request.json()
    const validation = createSubmissionSchema.safeParse(body)

    if (!validation.success) {
      return NextResponse.json(
        { error: 'Validation failed', details: validation.error.issues },
        { status: 400 }
      )
    }

    const { campaignId, videoUrl, platform } = validation.data

    // Validate Social Link ONLY if it's not an internal upload
    const isInternalUpload = videoUrl.startsWith('/uploads/') || videoUrl.includes('s3') || videoUrl.includes('swiplay-assets')

    if (!isInternalUpload) {
      const { validateSocialLink } = await import('@/lib/social-validation')
      if (!validateSocialLink(videoUrl, platform)) {
        return NextResponse.json(
          { error: `Invalid ${platform} URL format` },
          { status: 400 }
        )
      }
    }

    // Verify campaign exists and is active
    const campaign = await db.query.campaigns.findFirst({
      where: and(
        eq(campaigns.id, campaignId),
        eq(campaigns.status, 'active')
      ),
    })

    if (!campaign) {
      return NextResponse.json(
        { error: 'Campaign not found or not active' },
        { status: 404 }
      )
    }

    // Check if creator already submitted to this campaign
    const existingSubmission = await db.query.submissions.findFirst({
      where: and(
        eq(submissions.campaignId, campaignId),
        eq(submissions.creatorId, authResult.userId!)
      ),
    })

    if (existingSubmission) {
      return NextResponse.json(
        { error: 'You have already submitted to this campaign' },
        { status: 409 }
      )
    }

    const [newSubmission] = await db
      .insert(submissions)
      .values({
        campaignId,
        creatorId: authResult.userId!,
        videoUrl,
        platform,
        status: 'pending',
        submittedAt: new Date(),
        createdAt: new Date(),
        updatedAt: new Date(),
      })
      .returning()

    // Discord Log
    const { discord } = await import('@/lib/discord-logger')
    await discord.submission(
      'Nouvelle Soumission',
      `**${user.name || user.email}** a soumis une vidéo pour **${campaign.title}**`,
      [
        { name: 'Creator', value: user.name || 'Unknown', inline: true },
        { name: 'Campaign', value: campaign.title, inline: true },
        { name: 'Platform', value: platform, inline: true },
        { name: 'Video URL', value: videoUrl.substring(0, 100) }
      ]
    )

    return NextResponse.json(newSubmission, { status: 201 })
  } catch (error) {
    console.error('Create submission error:', error)
    return NextResponse.json(
      { error: 'Failed to create submission' },
      { status: 500 }
    )
  }
}