import { NextResponse } from 'next/server'
import { auth } from '@/auth'
import { db } from '@/lib/db'
import { users } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'

// GET current user's locale preference
export async function GET() {
    try {
        const session = await auth()
        if (!session?.user) {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
        }

        const user = await db.query.users.findFirst({
            where: eq(users.id, session.user.id),
            columns: { preferredLocale: true }
        })

        return NextResponse.json({ locale: user?.preferredLocale || 'fr' })
    } catch (error) {
        console.error('Get locale error:', error)
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}

// POST to update locale preference
export async function POST(req: Request) {
    try {
        const session = await auth()
        if (!session?.user) {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
        }

        const { locale } = await req.json()

        if (!locale || !['fr', 'en'].includes(locale)) {
            return NextResponse.json({ error: 'Invalid locale' }, { status: 400 })
        }

        await db.update(users)
            .set({ preferredLocale: locale })
            .where(eq(users.id, session.user.id))

        return NextResponse.json({ success: true, locale })
    } catch (error) {
        console.error('Update locale error:', error)
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}
