import NextAuth from "next-auth"
import { DrizzleAdapter } from "@auth/drizzle-adapter"
import { db } from "@/lib/db"
import { authConfig } from "./auth.config"
import Nodemailer from "next-auth/providers/nodemailer"
import Credentials from "next-auth/providers/credentials"
import { loginCodes, users, accounts, sessions, verificationTokens } from "@/lib/db/schema"
import { eq, and, gt, isNull } from "drizzle-orm"
import { sendDiscordNotification } from "@/lib/discord"
import { authRateLimiter } from "@/lib/rate-limit-db"

export const { handlers, auth, signIn, signOut } = NextAuth({
    adapter: DrizzleAdapter(db, {
        usersTable: users,
        accountsTable: accounts,
        sessionsTable: sessions,
        verificationTokensTable: verificationTokens,
    }),
    session: { strategy: "jwt" },
    ...authConfig,
    trustHost: true,
    debug: true,

    callbacks: {
        async jwt({ token, account, user, trigger, session }) {
            if (account) {
                token.accessToken = account.access_token
                token.refreshToken = account.refresh_token
            }
            if (user) {
                token.role = user.role
                token.id = user.id
                token.onboardingComplete = user.onboardingComplete
                token.company = user.company
                token.isApproved = user.isApproved
            }

            if (trigger === "update") {
                // Handle session updates (from client-side update() call)
                if (session?.image) {
                    token.picture = session.image
                }
                if (session?.onboardingComplete !== undefined) {
                    token.onboardingComplete = session.onboardingComplete
                }
            }

            return token
        },
        async session({ session, token }) {
            if (token && session.user) {
                session.user.role = token.role as string
                session.user.id = token.id as string
                session.user.onboardingComplete = token.onboardingComplete as number
                session.user.company = token.company as string
                session.user.isApproved = token.isApproved as number
                session.accessToken = token.accessToken as string
                session.refreshToken = token.refreshToken as string
            }
            return session
        }
    },
    providers: [
        Credentials({
            name: "OTP",
            credentials: {
                email: { label: "Email", type: "email" },
                code: { label: "Code", type: "text" },
                role: { label: "Role", type: "text" }
            },
            async authorize(credentials) {
                const email = (credentials.email as string).toLowerCase().trim()
                const code = (credentials.code as string).trim()
                const role = credentials.role as string

                if (!email || !code || !role) return null

                // Rate Limit by Email (5 attempts per 15 mins)
                if (!await authRateLimiter.check(5, email)) {
                    console.warn(`[AUTH_RATE_LIMIT] ${email} blocked`)
                    return null
                }

                // Find valid code
                const validCode = await db.query.loginCodes.findFirst({
                    where: and(
                        eq(loginCodes.email, email),
                        eq(loginCodes.code, code),
                        eq(loginCodes.role, role as any),
                        gt(loginCodes.expiresAt, new Date()),
                        isNull(loginCodes.usedAt)
                    )
                })

                if (!validCode) return null

                // Mark code as used
                await db.update(loginCodes)
                    .set({ usedAt: new Date() })
                    .where(eq(loginCodes.id, validCode.id))

                // Get user
                let user
                if (email === 'eyzuro@swiplay.com' || role === 'admin') {
                    user = await db.query.users.findFirst({
                        where: eq(users.email, email)
                    })
                } else {
                    user = await db.query.users.findFirst({
                        where: and(
                            eq(users.email, email),
                            eq(users.role, role as any)
                        )
                    })
                }

                // Create user if not exists (for non-admin)
                if (!user && role !== 'admin') {
                    await db.insert(users).values({
                        email,
                        role: role as any,
                        isApproved: 0,
                        createdAt: new Date(),
                        updatedAt: new Date(),
                    })

                    user = await db.query.users.findFirst({
                        where: and(eq(users.email, email), eq(users.role, role as any))
                    })

                    await sendDiscordNotification(
                        '👤 Nouvel Utilisateur',
                        `Un nouvel utilisateur **${email}** s'est inscrit en tant que **${role}**.`,
                        'success'
                    )
                }

                if (!user) return null

                await sendDiscordNotification(
                    '🔑 Connexion',
                    `L'utilisateur **${email}** (${role}) s'est connecté.`,
                    'info'
                )

                return {
                    id: user.id,
                    email: user.email,
                    name: user.name,
                    image: user.image,
                    role: user.role,
                    onboardingComplete: user.onboardingComplete,
                    isApproved: user.isApproved,
                }
            }
        }),
        Nodemailer({
            server: {
                host: process.env.EMAIL_SERVER_HOST,
                port: Number(process.env.EMAIL_SERVER_PORT),
                auth: {
                    user: process.env.EMAIL_SERVER_USER,
                    pass: process.env.EMAIL_SERVER_PASSWORD,
                },
            },
            from: process.env.EMAIL_FROM,
        }),
    ],
})
