'use client'

import { useState } from 'react'
import Link from 'next/link'
import Image from 'next/image'
import { Button } from '@/components/ui'
import { Globe, ChevronDown } from 'lucide-react'
import { useRouter } from 'next/navigation'

interface LandingHeaderProps {
    locale?: string
    view?: 'creator' | 'studio'
}

const translations = {
    fr: {
        nav: { login: 'Connexion' },
        cta: {
            button_creator: 'Débuter',
            button_studio: 'Débuter'
        }
    },
    en: {
        nav: { login: 'Login' },
        cta: {
            button_creator: 'Start Now',
            button_studio: 'Start Now'
        }
    }
}

export function LandingHeader({ locale = 'fr', view = 'creator' }: LandingHeaderProps) {
    const [showLangMenu, setShowLangMenu] = useState(false)
    const router = useRouter()
    const t = translations[locale as keyof typeof translations] || translations.fr

    const switchLocale = (newLocale: string) => {
        router.push(`/${newLocale}`)
        setShowLangMenu(false)
    }

    return (
        <nav className="fixed top-0 left-0 right-0 z-50 border-b border-white/10 bg-black/80 backdrop-blur-xl">
            <div className="max-w-6xl mx-auto px-4 md:px-6 h-14 md:h-16 flex items-center justify-between">
                <Link href="/" className="flex items-center">
                    <span className="text-xl md:text-2xl font-bold tracking-tight">SWIPL</span>
                    <Image src="/swiplay-logo.png" alt="A" width={24} height={24} className="inline -mx-1 md:-mx-1.5 md:w-[32px] md:h-[32px]" />
                    <span className="text-xl md:text-2xl font-bold tracking-tight">Y</span>
                </Link>
                <div className="flex items-center gap-2 md:gap-4">
                    {/* Language Switcher */}
                    <div className="relative">
                        <button
                            onClick={() => setShowLangMenu(!showLangMenu)}
                            className="flex items-center gap-1 px-2 md:px-3 py-1.5 rounded-full bg-white/5 hover:bg-white/10 transition-colors text-sm"
                        >
                            <Globe className="h-4 w-4 hidden md:block" />
                            {locale === 'fr' ? '🇫🇷' : '🇬🇧'}
                            <ChevronDown className="h-3 w-3" />
                        </button>
                        {showLangMenu && (
                            <div className="absolute top-full right-0 mt-2 py-1 rounded-xl bg-zinc-900 border border-white/10 shadow-xl min-w-[120px]">
                                <button onClick={() => switchLocale('fr')} className={`w-full px-4 py-2 text-left text-sm hover:bg-white/5 flex items-center gap-2 ${locale === 'fr' ? 'text-purple-400' : ''}`}>
                                    🇫🇷 Français
                                </button>
                                <button onClick={() => switchLocale('en')} className={`w-full px-4 py-2 text-left text-sm hover:bg-white/5 flex items-center gap-2 ${locale === 'en' ? 'text-purple-400' : ''}`}>
                                    🇬🇧 English
                                </button>
                            </div>
                        )}
                    </div>
                    <Button variant="ghost" className="hidden md:flex text-white hover:text-white hover:bg-white/10" asChild>
                        <Link href="/auth">{t.nav.login}</Link>
                    </Button>
                    <Button size="sm" className="bg-white text-black hover:bg-white/90 text-xs md:text-sm px-3 md:px-4" asChild>
                        <Link href="/auth">
                            {view === 'creator' ? t.cta.button_creator : t.cta.button_studio}
                        </Link>
                    </Button>
                </div>
            </div>
        </nav>
    )
}
