'use client'

import { useState, useEffect } from 'react'
import { Button } from '@/components/ui'
import { ArrowRight, ChevronRight, ChevronDown, Instagram, Globe } from 'lucide-react'
import Link from 'next/link'
import Image from 'next/image'
import { useParams, useRouter } from 'next/navigation'
import { StudioStep1Builder } from './StudioStep1Builder'
import { StudioStep2Validation } from './StudioStep2Validation'

const PARTNER_LOGOS = [
    "ankama_blanc.png",
    "gameforge_blanc.png",
    "magic_blanc.png",
    "twofrogs_blanc.png"
]

const CREATORS = [
    {
        name: 'Eyzuro',
        avatar: '/creators/eyzuro.png',
        followers: '+230k',
        socials: {
            tiktok: 'https://www.tiktok.com/@eyzuro',
            youtube: 'https://www.youtube.com/@eyzuro',
            instagram: 'https://www.instagram.com/eyzuro'
        }
    },
    {
        name: 'Rifflop',
        avatar: '/creators/rifflop.png',
        followers: '+280k',
        socials: {
            tiktok: 'https://www.tiktok.com/@rifflop',
            youtube: 'https://www.youtube.com/@RiffleStudio',
            instagram: 'https://www.instagram.com/rifflop_'
        }
    },
]

// Translations
const translations = {
    fr: {
        hero: {
            title_studio: 'Transformez les swipes',
            title_studio_highlight: 'en joueurs.',
            subtitle_studio: 'Connectez votre jeu à notre réseau et ne payez que les vidéos qui génèrent des vues.',
            cta_studio: 'Lancer ma campagne',
            title_creator: 'Arrête de poster',
            title_creator_highlight: 'gratuitement.',
            subtitle_creator: 'Connecte-toi aux studios et sois payé pour chaque vue sur les jeux que tu choisis.',
            cta_creator: 'Commencer à gagner',
        },
        nav: { login: 'Connexion' },
        trust: 'Ils nous font confiance',
        comparison: {
            creator: {
                title: 'Pourquoi choisir',
                title_highlight: 'Swiplay',
                headers: { criteria: 'VS', platforms: 'Fonds Créateurs', swiplay: 'Swiplay' },
                rows: [
                    { criteria: 'Revenu / 1M vues', platforms: '< 100 €', swiplay: '4 000 €+' },
                    { criteria: 'RPM (1k vues)', platforms: '< 0,10 €', swiplay: 'Minimum 4€' },
                    { criteria: 'Vues payées', platforms: 'Aléatoire', swiplay: '100% Garanties' },
                    { criteria: 'Monétisation', platforms: '1 Plateforme', swiplay: 'Multi-plateformes' },
                ]
            },
            studio: {
                title: 'Pourquoi choisir',
                title_highlight: 'Swiplay',
                headers: { criteria: 'VS', platforms: 'Agences classiques', swiplay: 'Swiplay' },
                rows: [
                    { criteria: 'Modèle de paiement', platforms: 'Forfait fixe', swiplay: 'Paiement à la vue' },
                    { criteria: 'Validation contenu', platforms: 'Limité', swiplay: 'Validation pré-publication' },
                    { criteria: 'Créateurs', platforms: 'Imposés', swiplay: 'Réseau gaming ciblé' },
                    { criteria: 'Transparence', platforms: 'Opaque', swiplay: 'Analytics temps réel' },
                ]
            }
        },
        howItWorks: {
            title: 'Comment ça',
            title_highlight: 'marche',
            creator: [
                { title: '1. Choisis', desc: 'Sélectionne les campagnes qui te plaisent directement dans le catalogue.' },
                { title: '2. Poste', desc: 'Publie ta vidéo sur tes réseaux.' },
                { title: '3. Encaisse', desc: 'Reçois ton paiement automatiquement pour chaque vue générée.' },
            ],
            studio: [
                { title: '1. Définissez', desc: 'Créez votre campagne, fixez votre budget et votre prix par vue.' },
                { title: '2. Validez', desc: 'Approuvez chaque vidéo avant sa publication par nos créateurs.' },
                { title: '3. Payez', desc: 'Ne payez que les vues réellement générées sur vos contenus.' },
            ]
        },
        elite: {
            title_creator: 'Rejoignez',
            title_highlight_creator: 'l\'élite',
            title_studio: 'Travaillez avec',
            title_highlight_studio: 'l\'élite des créateurs',
            subtitle_studio: 'Découvrez les talents qui performent déjà sur Swiplay.'
        },
        cta: {
            title_creator: 'Prêt à commencer ?',
            subtitle_creator: 'Fais de ta passion un vrai revenu.',
            button_creator: 'Créer un compte',
            title_studio: 'Lancez votre campagne',
            subtitle_studio: 'Accédez à des créateurs qui performent.',
            button_studio: 'Demander un accès'
        },
        footer: { rights: 'Tous droits réservés.', terms: 'CGU', privacy: 'Confidentialité', legal: 'Mentions légales' }
    },
    en: {
        hero: {
            title_studio: 'Turn Swipes',
            title_studio_highlight: 'into Players.',
            subtitle_studio: 'Connect with our network and pay only for generated views.',
            cta_studio: 'Launch my campaign',
            title_creator: 'Stop Posting',
            title_creator_highlight: 'for Free.',
            subtitle_creator: 'Connect with studios and get paid for every view on the games you choose.',
            cta_creator: 'Start Earning',
            nav_button_creator: 'Start Now',
            nav_button_studio: 'Start Now',
        },
        nav: { login: 'Login' },
        trust: 'They Trust Us',
        comparison: {
            creator: {
                title: 'Why choose',
                title_highlight: 'Swiplay',
                headers: { criteria: 'VS', platforms: 'Creator Funds', swiplay: 'Swiplay' },
                rows: [
                    { criteria: 'Revenue / 1M views', platforms: '< €100', swiplay: '€4,000+' },
                    { criteria: 'RPM (1k views)', platforms: '< €0.10', swiplay: 'Minimum €4' },
                    { criteria: 'Paid Views', platforms: 'Random', swiplay: '100% Guaranteed' },
                    { criteria: 'Monetization', platforms: '1 Platform', swiplay: 'Multi-platform' },
                ]
            },
            studio: {
                title: 'Why choose',
                title_highlight: 'Swiplay',
                headers: { criteria: 'VS', platforms: 'Traditional Agencies', swiplay: 'Swiplay' },
                rows: [
                    { criteria: 'Payment Model', platforms: 'Fixed fee', swiplay: 'Pay per view' },
                    { criteria: 'Content Validation', platforms: 'Limited', swiplay: 'Pre-publication approval' },
                    { criteria: 'Creators', platforms: 'Imposed', swiplay: 'Gaming-focused network' },
                    { criteria: 'Transparency', platforms: 'Opaque', swiplay: 'Real-time analytics' },
                ]
            }
        },
        howItWorks: {
            title: 'How It',
            title_highlight: 'Works',
            creator: [
                { title: '1. Pick', desc: 'Select campaigns that match your style from the marketplace.' },
                { title: '2. Post', desc: 'Publish your video on your channels.' },
                { title: '3. Earn', desc: 'Get paid automatically for every view generated.' },
            ],
            studio: [
                { title: '1. Define', desc: 'Create your campaign, set your budget and price per view.' },
                { title: '2. Validate', desc: 'Approve each video before publication by our creators.' },
                { title: '3. Pay', desc: 'Only pay for views actually generated on your content.' },
            ]
        },
        elite: {
            title_creator: 'Join the',
            title_highlight_creator: 'Elite Creators',
            title_studio: 'Work with the',
            title_highlight_studio: 'Elite Creators',
            subtitle_studio: 'Discover current top performing talents on Swiplay.'
        },
        cta: {
            title_creator: 'Ready to start?',
            subtitle_creator: 'Join creators who monetize their gaming passion.',
            button_creator: 'Create an account',
            title_studio: 'Launch your campaign',
            subtitle_studio: 'Connect with our network of verified gaming creators.',
            button_studio: 'Request access'
        },
        footer: { rights: 'All rights reserved.', terms: 'Terms', privacy: 'Privacy', legal: 'Legal' }
    }
}

export interface LandingPageProps {
    defaultView?: 'creator' | 'studio'
}

function LandingPage({ defaultView = 'creator' }: LandingPageProps) {
    const params = useParams()
    const router = useRouter()
    const locale = (params?.locale as string) || 'fr'
    const t = translations[locale as keyof typeof translations] || translations.fr

    const [view, setView] = useState<'creator' | 'studio'>(defaultView)
    const [showLangMenu, setShowLangMenu] = useState(false)

    const switchLocale = (newLocale: string) => {
        router.push(`/${newLocale}`)
        setShowLangMenu(false)
    }

    // Apple-style scroll reveal animations
    useEffect(() => {
        const observer = new IntersectionObserver(
            (entries) => {
                entries.forEach((entry) => {
                    if (entry.isIntersecting) {
                        entry.target.classList.add('animate-in')
                    }
                })
            },
            { threshold: 0.1, rootMargin: '0px 0px -50px 0px' }
        )

        document.querySelectorAll('.reveal, .reveal-left, .reveal-right').forEach((el) => observer.observe(el))
        return () => observer.disconnect()
    }, [])

    return (
        <div className="min-h-screen overflow-hidden bg-black text-white">
            {/* Animation styles - Progressive scroll reveal */}
            <style jsx global>{`
                .reveal {
                    opacity: 0;
                    transform: translateY(80px) scale(0.95);
                    transition: opacity 1s cubic-bezier(0.16, 1, 0.3, 1), 
                                transform 1.2s cubic-bezier(0.16, 1, 0.3, 1);
                }
                .reveal.animate-in {
                    opacity: 1;
                    transform: translateY(0) scale(1);
                }
                .reveal-delay-1 { transition-delay: 0.15s; }
                .reveal-delay-2 { transition-delay: 0.3s; }
                .reveal-delay-3 { transition-delay: 0.45s; }
                .reveal-delay-4 { transition-delay: 0.6s; }
                
                /* Fade from sides */
                .reveal-left {
                    opacity: 0;
                    transform: translateX(-60px);
                    transition: opacity 1s cubic-bezier(0.16, 1, 0.3, 1), 
                                transform 1.2s cubic-bezier(0.16, 1, 0.3, 1);
                }
                .reveal-right {
                    opacity: 0;
                    transform: translateX(60px);
                    transition: opacity 1s cubic-bezier(0.16, 1, 0.3, 1), 
                                transform 1.2s cubic-bezier(0.16, 1, 0.3, 1);
                }
                .reveal-left.animate-in,
                .reveal-right.animate-in {
                    opacity: 1;
                    transform: translateX(0);
                }
                /* Scroll animation for logos - exact from backup */
                @keyframes scroll-left {
                    0% { transform: translateX(0); }
                    100% { transform: translateX(-50%); }
                }
                .animate-scroll-left {
                    animation: scroll-left 40s linear infinite;
                }
                .mask-linear-gradient {
                    mask-image: linear-gradient(to right, transparent, black 10%, black 90%, transparent);
                    -webkit-mask-image: linear-gradient(to right, transparent, black 10%, black 90%, transparent);
                }
                .mask-radial-faded {
                    mask-image: radial-gradient(ellipse at center, black 40%, transparent 70%);
                    -webkit-mask-image: radial-gradient(ellipse at center, black 40%, transparent 70%);
                }
            `}</style>
            {/* Navbar - Apple style */}
            <nav className="fixed top-0 left-0 right-0 z-50 border-b border-white/10 bg-black/80 backdrop-blur-xl">
                <div className="max-w-6xl mx-auto px-4 md:px-6 h-14 md:h-16 flex items-center justify-between">
                    <Link href="/" className="flex items-center">
                        <span className="text-lg md:text-xl font-bold tracking-tight">SWIPL</span>
                        <Image src="/swiplay-logo.png" alt="A" width={20} height={20} className="inline -mx-0.5 md:w-[24px] md:h-[24px]" />
                        <span className="text-lg md:text-xl font-bold tracking-tight">Y</span>
                    </Link>
                    <div className="flex items-center gap-2 md:gap-4">
                        {/* Language Switcher */}
                        <div className="relative">
                            <button
                                onClick={() => setShowLangMenu(!showLangMenu)}
                                className="flex items-center gap-1 px-2 md:px-3 py-1.5 rounded-full bg-white/5 hover:bg-white/10 transition-colors text-sm"
                            >
                                <Globe className="h-4 w-4 hidden md:block" />
                                {locale === 'fr' ? '🇫🇷' : '🇬🇧'}
                                <ChevronDown className="h-3 w-3" />
                            </button>
                            {showLangMenu && (
                                <div className="absolute top-full right-0 mt-2 py-1 rounded-xl bg-zinc-900 border border-white/10 shadow-xl min-w-[120px]">
                                    <button onClick={() => switchLocale('fr')} className={`w-full px-4 py-2 text-left text-sm hover:bg-white/5 flex items-center gap-2 ${locale === 'fr' ? 'text-purple-400' : ''}`}>
                                        🇫🇷 Français
                                    </button>
                                    <button onClick={() => switchLocale('en')} className={`w-full px-4 py-2 text-left text-sm hover:bg-white/5 flex items-center gap-2 ${locale === 'en' ? 'text-purple-400' : ''}`}>
                                        🇬🇧 English
                                    </button>
                                </div>
                            )}
                        </div>
                        <Button variant="ghost" className="hidden md:flex text-white hover:text-white hover:bg-white/10" asChild>
                            <Link href="/auth">{t.nav.login}</Link>
                        </Button>
                        <Button size="sm" className="bg-white text-black hover:bg-white/90 text-xs md:text-sm px-3 md:px-4" asChild>
                            <Link href="/auth">{locale === 'fr' ? 'Commencer' : 'Start Now'}</Link>
                        </Button>
                    </div>
                </div>
            </nav>

            {/* 1. Hero Section */}
            <section className="relative pt-32 pb-24 px-6 overflow-hidden">
                {/* Gradient orbs */}
                <div className="absolute top-20 left-1/4 w-96 h-96 bg-purple-600/30 rounded-full blur-3xl animate-pulse" />
                <div className="absolute top-40 right-1/4 w-80 h-80 bg-blue-600/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '1s' }} />

                <div className="relative z-10 max-w-5xl mx-auto">
                    {/* Toggle */}
                    <div className="flex justify-center mb-8 md:mb-12">
                        <div className="inline-flex p-1 rounded-full bg-white/5 border border-white/10">
                            <button
                                onClick={() => setView('creator')}
                                className={`px-5 md:px-8 py-2.5 md:py-3 rounded-full text-sm font-medium transition-all duration-300 ${view === 'creator' ? 'bg-white text-black' : 'text-white/60 hover:text-white'
                                    }`}
                            >
                                {locale === 'fr' ? 'Créateur' : 'Creator'}
                            </button>
                            <button
                                onClick={() => setView('studio')}
                                className={`px-5 md:px-8 py-2.5 md:py-3 rounded-full text-sm font-medium transition-all duration-300 ${view === 'studio' ? 'bg-white text-black' : 'text-white/60 hover:text-white'
                                    }`}
                            >
                                Studio
                            </button>
                        </div>
                    </div>

                    <div className="text-center">
                        <h1 className="text-4xl md:text-5xl lg:text-7xl font-bold tracking-tight mb-4 md:mb-6 leading-tight">
                            {view === 'creator' ? t.hero.title_creator : t.hero.title_studio}<br />
                            <span className="bg-gradient-to-r from-purple-400 via-pink-400 to-blue-400 text-transparent bg-clip-text">
                                {view === 'creator' ? t.hero.title_creator_highlight : t.hero.title_studio_highlight}
                            </span>
                        </h1>

                        <p className="text-base md:text-xl text-white/60 max-w-2xl mx-auto mb-8 md:mb-10 px-4 md:px-0">
                            {view === 'creator' ? t.hero.subtitle_creator : t.hero.subtitle_studio}
                        </p>

                        <Button size="lg" className="px-6 md:px-10 h-12 md:h-14 text-sm md:text-base bg-white text-black hover:bg-gray-100 border-0" asChild>
                            <Link href="/auth">
                                {view === 'creator' ? t.hero.cta_creator : t.hero.cta_studio}
                                <ArrowRight className="h-4 w-4 ml-2" />
                            </Link>
                        </Button>
                    </div>
                </div>
            </section>

            {/* 4. Trust Section - Exact from backup with dark bg and reduced margins */}
            <section className="py-8 bg-zinc-900/50 border-y border-white/10">
                <div className="max-w-3xl mx-auto overflow-hidden relative mask-linear-gradient">
                    <p className="text-center text-sm text-white/30 uppercase tracking-widest mb-6 font-medium px-6">{t.trust}</p>
                    <div className="flex w-max animate-scroll-left gap-12 md:gap-24 items-center">
                        {/* Double the logos for seamless loop */}
                        {[...PARTNER_LOGOS, ...PARTNER_LOGOS, ...PARTNER_LOGOS, ...PARTNER_LOGOS].map((logo, i) => (
                            <div key={i} className="relative h-8 w-24 md:h-10 md:w-32 flex-shrink-0 transition-opacity duration-300 opacity-40 hover:opacity-100">
                                <Image src={`/images/${logo}`} alt="Partner" fill className="object-contain" />
                            </div>
                        ))}
                    </div>
                </div>
            </section>

            {/* 3. How it Works - Each step with its own visual */}
            <section className="py-24 px-6 bg-gradient-to-b from-transparent via-purple-900/10 to-transparent">
                <div className="max-w-6xl mx-auto">
                    <div className="reveal text-center mb-20">
                        <h2 className="text-3xl md:text-5xl font-bold mb-4">
                            {t.howItWorks.title} <span className="text-transparent bg-clip-text bg-gradient-to-r from-purple-400 to-blue-500">{t.howItWorks.title_highlight}</span> ?
                        </h2>
                    </div>

                    <div className="space-y-24">
                        {/* Step 1 - Choose Campaign */}
                        <div className="reveal grid md:grid-cols-2 gap-12 items-center">
                            <div className="order-2 md:order-1">
                                <div className="flex items-center gap-4 mb-6">
                                    <div className="w-14 h-14 rounded-2xl bg-gradient-to-br from-purple-500 to-pink-500 flex items-center justify-center font-bold text-xl shadow-lg shadow-purple-500/30">1</div>
                                    <h3 className="text-2xl font-bold">{t.howItWorks[view][0].title}</h3>
                                </div>
                                <p className="text-lg text-white/60 leading-relaxed">{t.howItWorks[view][0].desc}</p>
                            </div>
                            <div className="order-1 md:order-2">
                                <div className="relative">
                                    <div className="absolute inset-0 bg-purple-500/20 rounded-3xl blur-2xl" />
                                    {view === 'creator' ? (
                                        /* Creator: Campaign Card - Exact from backup */
                                        <div className="relative rounded-2xl bg-zinc-900/90 backdrop-blur-xl border border-white/10 shadow-2xl overflow-hidden transform hover:scale-[1.02] transition-all duration-500 group">
                                            <div className="absolute inset-0">
                                                <Image src="/images/campaigns/underarock_1.jpg" alt="Under a Rock" fill className="object-cover transition-transform duration-700 group-hover:scale-110" />
                                                <div className="absolute inset-0 bg-gradient-to-t from-black via-black/90 to-black/20" />
                                            </div>
                                            <div className="relative p-6 pt-32">
                                                <div className="flex justify-between items-start mb-4">
                                                    <span className="px-3 py-1 rounded-full bg-blue-500/20 text-blue-400 text-xs font-semibold backdrop-blur-sm border border-blue-500/10">CPM</span>
                                                    <div className="text-right">
                                                        <p className="text-2xl font-bold text-white">€12.00</p>
                                                        <p className="text-xs text-white/50">per 1k views</p>
                                                    </div>
                                                </div>
                                                <h4 className="text-2xl font-bold text-white mb-2">Under a Rock</h4>
                                                <div className="space-y-4">
                                                    <div>
                                                        <div className="flex justify-between text-sm mb-2">
                                                            <span className="text-white/70">Budget left</span>
                                                            <span className="text-white font-medium">€8,500</span>
                                                        </div>
                                                        <div className="w-full h-2 bg-white/10 rounded-full overflow-hidden">
                                                            <div className="h-full bg-gradient-to-r from-purple-500 to-blue-500 w-[85%]" />
                                                        </div>
                                                    </div>
                                                    <button className="w-full py-3 rounded-xl bg-white text-black font-bold hover:bg-gray-100 transition-colors flex items-center justify-center gap-2">
                                                        Apply Now
                                                        <ArrowRight className="w-4 h-4" />
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    ) : (
                                        /* Studio: Campaign Builder Refactored */
                                        <StudioStep1Builder locale={locale} />
                                    )}
                                </div>
                            </div>
                        </div>

                        {/* Step 2 - Post Content */}
                        <div className="reveal grid md:grid-cols-2 gap-12 items-center">
                            <div>
                                <div className="relative">
                                    <div className="absolute inset-0 bg-white/5 rounded-3xl blur-2xl" />
                                    {view === 'creator' ? (
                                        /* Creator: Video with radial mask bg - Exact from backup */
                                        <div className="relative">
                                            <div className="absolute inset-0 rounded-3xl overflow-hidden mask-radial-faded">
                                                <Image src="/images/campaigns/underarock_2.jpg" alt="Background" fill className="object-cover opacity-60 scale-105" />
                                                <div className="absolute inset-0 bg-gradient-to-t from-zinc-900 via-zinc-900/40 to-transparent" />
                                            </div>
                                            <div className="relative rounded-2xl bg-zinc-900/90 backdrop-blur-xl border border-white/10 shadow-2xl overflow-hidden transform hover:scale-[1.02] transition-all duration-500 max-w-[200px] mx-auto">
                                                <video autoPlay loop muted playsInline className="w-full aspect-[9/16] object-cover rounded-2xl">
                                                    <source src="/videos/eyzuro_underarock.mp4" type="video/mp4" />
                                                </video>
                                                <div className="absolute bottom-0 left-0 right-0 p-4 bg-gradient-to-t from-black/80 to-transparent">
                                                    <div className="flex items-center gap-3">
                                                        <div className="w-10 h-10 rounded-full overflow-hidden ring-2 ring-purple-500/50">
                                                            <Image src="/creators/eyzuro.png" alt="Eyzuro" width={40} height={40} className="w-full h-full object-cover" />
                                                        </div>
                                                        <div>
                                                            <p className="font-semibold text-white text-sm">Eyzuro</p>
                                                            <p className="text-xs text-white/50">Under a Rock • Gameforge</p>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    ) : (
                                        /* Studio: Validation with approval UI */
                                        <StudioStep2Validation locale={locale} />
                                    )}
                                </div>
                            </div>
                            <div>
                                <div className="flex items-center gap-4 mb-6">
                                    <div className="w-14 h-14 rounded-2xl bg-gradient-to-br from-pink-500 to-orange-500 flex items-center justify-center font-bold text-xl shadow-lg shadow-pink-500/30">2</div>
                                    <h3 className="text-2xl font-bold">{t.howItWorks[view][1].title}</h3>
                                </div>
                                <p className="text-lg text-white/60 leading-relaxed">{t.howItWorks[view][1].desc}</p>
                            </div>
                        </div>

                        {/* Step 3 - Get Paid */}
                        <div className="reveal grid md:grid-cols-2 gap-12 items-center">
                            <div className="order-2 md:order-1">
                                <div className="flex items-center gap-4 mb-6">
                                    <div className="w-14 h-14 rounded-2xl bg-gradient-to-br from-blue-500 to-cyan-500 flex items-center justify-center font-bold text-xl shadow-lg shadow-blue-500/30">3</div>
                                    <h3 className="text-2xl font-bold">{t.howItWorks[view][2].title}</h3>
                                </div>
                                <p className="text-lg text-white/60 leading-relaxed">{t.howItWorks[view][2].desc}</p>
                            </div>
                            <div className="order-1 md:order-2">
                                <div className="relative">
                                    <div className="absolute inset-0 bg-emerald-500/20 rounded-3xl blur-2xl" />
                                    {view === 'creator' ? (
                                        /* Creator: Wallet UI - Exact from backup */
                                        <div className="relative p-8 rounded-3xl bg-zinc-900/90 backdrop-blur-xl border border-white/10 shadow-2xl overflow-hidden group hover:scale-[1.02] transition-transform duration-500">
                                            <div className="text-center mb-8">
                                                <div className="w-16 h-16 mx-auto bg-white/5 rounded-full flex items-center justify-center mb-4 border border-white/10">
                                                    <svg fill="none" viewBox="0 0 24 24" stroke="currentColor" className="w-8 h-8 text-emerald-400">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                                    </svg>
                                                </div>
                                                <p className="text-sm font-medium text-white/50 mb-2 uppercase tracking-widest">{locale === 'fr' ? 'Solde disponible' : 'Available Balance'}</p>
                                                <h3 className="text-5xl font-bold text-white tracking-tight">€2,770.00</h3>
                                            </div>
                                            <div className="bg-white/5 rounded-2xl p-4 mb-8 border border-white/5 flex items-center justify-between">
                                                <div className="flex items-center gap-4">
                                                    <div className="w-10 h-10 rounded-lg overflow-hidden relative border border-white/10">
                                                        <Image src="/images/campaigns/underarock_3.jpg" alt="Logo" fill className="object-cover" />
                                                    </div>
                                                    <div>
                                                        <p className="font-bold text-white text-sm">Under a Rock</p>
                                                        <p className="text-xs text-white/50">{locale === 'fr' ? 'Campagne terminée' : 'Campaign completed'}</p>
                                                    </div>
                                                </div>
                                                <span className="font-bold text-emerald-400">+ €920.00</span>
                                            </div>
                                            <div className="space-y-3">
                                                <button className="w-full py-4 rounded-xl bg-white text-black font-bold hover:bg-gray-100 transition-colors shadow-lg flex items-center justify-center gap-2">
                                                    {locale === 'fr' ? 'Retirer' : 'Cashout'}
                                                    <ArrowRight className="w-4 h-4" />
                                                </button>
                                                <p className="text-center text-xs text-white/30">{locale === 'fr' ? 'Transfert sous 24h ouvrées' : 'Transfer within 24 business hours'}</p>
                                            </div>
                                        </div>
                                    ) : (
                                        /* Studio: Analytics with Jubily - Exact from backup */
                                        <div className="relative p-6 rounded-2xl bg-zinc-900/90 backdrop-blur-xl border border-white/10 shadow-2xl transform hover:scale-[1.02] transition-all duration-500">
                                            <div className="flex items-center justify-between mb-6">
                                                <h4 className="font-bold text-white text-lg">Under a Rock</h4>
                                                <span className="px-3 py-1 rounded-full bg-green-500/20 text-green-400 text-xs font-semibold">Active</span>
                                            </div>
                                            <div className="grid grid-cols-2 gap-4 mb-6">
                                                <div className="p-4 rounded-xl bg-white/5 border border-white/10">
                                                    <p className="text-xs text-white/50 mb-1">{locale === 'fr' ? 'Budget utilisé' : 'Budget Used'}</p>
                                                    <p className="text-xl font-bold text-white">€3,120<span className="text-sm font-normal text-white/40"> / €10,000</span></p>
                                                    <div className="w-full h-1 bg-white/10 rounded-full mt-2 overflow-hidden">
                                                        <div className="h-full bg-blue-500" style={{ width: '31%' }} />
                                                    </div>
                                                </div>
                                                <div className="p-4 rounded-xl bg-white/5 border border-white/10">
                                                    <p className="text-xs text-white/50 mb-1">{locale === 'fr' ? 'Vues totales' : 'Total Views'}</p>
                                                    <p className="text-xl font-bold text-white">385K</p>
                                                    <p className="text-xs text-green-400 mt-1 flex items-center gap-1">
                                                        <svg fill="none" viewBox="0 0 24 24" stroke="currentColor" className="w-3 h-3"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" /></svg>
                                                        +12% <span className="text-white/30 ml-1">{locale === 'fr' ? 'depuis hier' : 'since yesterday'}</span>
                                                    </p>
                                                </div>
                                            </div>
                                            <div>
                                                <p className="text-xs text-white/40 uppercase tracking-wider font-medium mb-3">{locale === 'fr' ? 'Top Créateurs' : 'Top Creators'}</p>
                                                <div className="space-y-3">
                                                    <div className="flex items-center justify-between p-2 rounded-lg hover:bg-white/5 transition-colors">
                                                        <div className="flex items-center gap-3">
                                                            <div className="w-8 h-8 rounded-full bg-white/10 overflow-hidden">
                                                                <Image src="/creators/eyzuro.png" alt="Eyzuro" width={32} height={32} className="object-cover" />
                                                            </div>
                                                            <div>
                                                                <p className="text-sm font-medium text-white">Eyzuro</p>
                                                                <p className="text-xs text-white/50">230K {locale === 'fr' ? 'vues' : 'views'}</p>
                                                            </div>
                                                        </div>
                                                        <div className="text-right">
                                                            <p className="text-sm font-bold text-white">€920</p>
                                                        </div>
                                                    </div>
                                                    <div className="flex items-center justify-between p-2 rounded-lg hover:bg-white/5 transition-colors">
                                                        <div className="flex items-center gap-3">
                                                            <div className="w-8 h-8 rounded-full bg-white/10 overflow-hidden">
                                                                <Image src="/creators/rifflop.png" alt="Rifflop" width={32} height={32} className="object-cover" />
                                                            </div>
                                                            <div>
                                                                <p className="text-sm font-medium text-white">Rifflop</p>
                                                                <p className="text-xs text-white/50">183K {locale === 'fr' ? 'vues' : 'views'}</p>
                                                            </div>
                                                        </div>
                                                        <div className="text-right">
                                                            <p className="text-sm font-bold text-white">€732</p>
                                                        </div>
                                                    </div>
                                                    <div className="flex items-center justify-between p-2 rounded-lg hover:bg-white/5 transition-colors">
                                                        <div className="flex items-center gap-3">
                                                            <div className="w-8 h-8 rounded-full bg-white/10 overflow-hidden">
                                                                <Image src="/creators/jubilygaming.png" alt="Jubily" width={32} height={32} className="object-cover" />
                                                            </div>
                                                            <div>
                                                                <p className="text-sm font-medium text-white">Jubily</p>
                                                                <p className="text-xs text-white/50">111K {locale === 'fr' ? 'vues' : 'views'}</p>
                                                            </div>
                                                        </div>
                                                        <div className="text-right">
                                                            <p className="text-sm font-bold text-white">€444</p>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </div>
                        </div >
                    </div >
                </div >
            </section >

            {/* 2. Comparison Table - Creator Only */}
            {
                view === 'creator' && (
                    <section className="py-24 px-6">
                        <div className="max-w-4xl mx-auto">
                            <div className="reveal text-center mb-16">
                                <h2 className="text-3xl md:text-5xl font-bold mb-6">
                                    {t.comparison[view].title} <span className="text-transparent bg-clip-text bg-gradient-to-r from-purple-400 to-blue-500">{t.comparison[view].title_highlight}</span> ?
                                </h2>
                            </div>

                            {/* Desktop Table */}
                            <div className="hidden md:block rounded-3xl overflow-hidden border border-white/10 bg-white/5 backdrop-blur-xl">
                                <table className="w-full text-left border-collapse">
                                    <thead>
                                        <tr className="bg-white/5">
                                            <th className="p-6 text-white/40 font-medium w-1/3">{t.comparison[view].headers.criteria}</th>
                                            <th className="p-6 text-white/40 font-medium w-1/3 text-center">{t.comparison[view].headers.platforms}</th>
                                            <th className="p-6 text-purple-400 font-bold text-lg w-1/3 text-center bg-purple-500/10">
                                                {t.comparison[view].headers.swiplay}
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-white/5">
                                        {t.comparison[view].rows.map((row, i) => (
                                            <tr key={i} className="hover:bg-white/5 transition-colors group">
                                                <td className="p-6 font-medium text-white/70 group-hover:text-white transition-colors">
                                                    {row.criteria}
                                                </td>
                                                <td className="p-6 text-white/40 text-center">
                                                    {row.platforms}
                                                </td>
                                                <td className="p-6 text-white font-bold text-lg text-center bg-purple-500/5 group-hover:bg-purple-500/10 transition-colors">
                                                    {(row.criteria === 'Monétisation' || row.criteria === 'Monetization') ? (
                                                        <div className="flex items-center justify-center gap-2">
                                                            <Instagram className="w-5 h-5 text-pink-500" />
                                                            <svg className="w-5 h-5 text-white" viewBox="0 0 24 24" fill="currentColor"><path d="M19.59 6.69a4.83 4.83 0 0 1-3.77-4.25V2h-3.45v13.67a2.89 2.89 0 0 1-5.2 1.74 2.89 2.89 0 0 1 2.31-4.64 2.93 2.93 0 0 1 .88.13V9.4a6.84 6.84 0 0 0-1-.05A6.33 6.33 0 0 0 5 20.1a6.34 6.34 0 0 0 10.86-4.43v-7a8.16 8.16 0 0 0 4.77 1.52v-3.4a4.85 4.85 0 0 1-1-.1z" /></svg>
                                                            <svg className="w-5 h-5 text-red-600" viewBox="0 0 24 24" fill="currentColor"><path d="M23.498 6.186a3.016 3.016 0 0 0-2.122-2.136C19.505 3.545 12 3.545 12 3.545s-7.505 0-9.377.505A3.017 3.017 0 0 0 .502 6.186C0 8.07 0 12 0 12s0 3.93.502 5.814a3.016 3.016 0 0 0 2.122 2.136c1.871.505 9.376.505 9.376.505s7.505 0 9.377-.505a3.015 3.015 0 0 0 2.122-2.136C24 15.93 24 12 24 12s0-3.93-.502-5.814zM9.545 15.568V8.432L15.818 12l-6.273 3.568z" /></svg>
                                                        </div>
                                                    ) : row.swiplay}
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>

                            {/* Mobile Cards */}
                            <div className="md:hidden space-y-4">
                                {t.comparison[view].rows.map((row, i) => (
                                    <div key={i} className="bg-white/5 border border-white/10 rounded-2xl p-5 relative overflow-hidden">
                                        <div className="absolute top-0 right-0 w-20 h-20 bg-purple-500/10 blur-2xl rounded-full -mr-10 -mt-10" />
                                        <h4 className="text-white/40 text-sm font-medium mb-4 uppercase tracking-wider">{row.criteria}</h4>
                                        <div className="flex justify-between items-center gap-4">
                                            <div className="flex-1">
                                                <div className="text-xs text-white/30 mb-1">{t.comparison[view].headers.platforms}</div>
                                                <div className="text-white/60 font-medium">{row.platforms}</div>
                                            </div>
                                            <div className="w-px h-10 bg-white/10" />
                                            <div className="flex-1 text-right">
                                                <div className="text-xs text-purple-400 mb-1 font-bold">Swiplay</div>
                                                <div className="text-white font-bold text-lg">{row.swiplay}</div>
                                            </div>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </div>
                    </section>
                )
            }

            {/* 5. Creators */}
            <section className="py-24 px-6">
                <div className="max-w-5xl mx-auto">
                    <div className="text-center mb-12">
                        <h2 className="text-3xl md:text-5xl font-bold mb-4">
                            {view === 'creator'
                                ? (locale === 'fr' ? 'Rejoignez ' : 'Join the ')
                                : (locale === 'fr' ? 'Travaillez avec ' : 'Work with the ')}
                            <span className="text-transparent bg-clip-text bg-gradient-to-r from-purple-400 to-blue-500">
                                {view === 'creator'
                                    ? (locale === 'fr' ? 'l\'élite des créateurs' : 'Elite Creators')
                                    : (locale === 'fr' ? 'l\'élite des créateurs' : 'Elite Creators')}
                            </span>
                        </h2>
                        <p className="text-white/60 text-lg md:text-xl max-w-3xl mx-auto">
                            {view === 'creator'
                                ? (locale === 'fr' ? 'Découvrez les talents qui performent déjà sur Swiplay.' : 'Discover current top performing talents on Swiplay.')
                                : t.elite.subtitle_studio}
                        </p>
                    </div>

                    <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
                        {CREATORS.map((creator, i) => (
                            <div key={i} className="group relative aspect-[3/4] rounded-3xl overflow-hidden">
                                <Image src={creator.avatar} alt={creator.name} fill className="object-cover transition-transform duration-700 group-hover:scale-110" />
                                <div className="absolute inset-0 bg-gradient-to-t from-black/90 via-black/20 to-transparent opacity-80 group-hover:opacity-100 transition-opacity" />
                                <div className="absolute bottom-0 left-0 right-0 p-4 md:p-6">
                                    <h3 className="text-sm md:text-xl font-bold mb-1">{creator.name}</h3>
                                    <div className="flex items-center justify-between">
                                        <span className="text-xs md:text-sm text-purple-300 font-medium">{creator.followers}</span>
                                        <div className="flex items-center gap-1.5">
                                            <a href={creator.socials.tiktok} target="_blank" className="w-6 h-6 md:w-8 md:h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-black hover:scale-110 transition-all">
                                                <svg className="w-3 h-3 md:w-4 md:h-4" viewBox="0 0 24 24" fill="currentColor"><path d="M19.59 6.69a4.83 4.83 0 0 1-3.77-4.25V2h-3.45v13.67a2.89 2.89 0 0 1-5.2 1.74 2.89 2.89 0 0 1 2.31-4.64 2.93 2.93 0 0 1 .88.13V9.4a6.84 6.84 0 0 0-1-.05A6.33 6.33 0 0 0 5 20.1a6.34 6.34 0 0 0 10.86-4.43v-7a8.16 8.16 0 0 0 4.77 1.52v-3.4a4.85 4.85 0 0 1-1-.1z" /></svg>
                                            </a>
                                            <a href={creator.socials.youtube} target="_blank" className="w-6 h-6 md:w-8 md:h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-600 hover:scale-110 transition-all">
                                                <svg className="w-3 h-3 md:w-4 md:h-4" viewBox="0 0 24 24" fill="currentColor"><path d="M23.498 6.186a3.016 3.016 0 0 0-2.122-2.136C19.505 3.545 12 3.545 12 3.545s-7.505 0-9.377.505A3.017 3.017 0 0 0 .502 6.186C0 8.07 0 12 0 12s0 3.93.502 5.814a3.016 3.016 0 0 0 2.122 2.136c1.871.505 9.376.505 9.376.505s7.505 0 9.377-.505a3.015 3.015 0 0 0 2.122-2.136C24 15.93 24 12 24 12s0-3.93-.502-5.814zM9.545 15.568V8.432L15.818 12l-6.273 3.568z" /></svg>
                                            </a>
                                            <a href={creator.socials.instagram} target="_blank" className="w-6 h-6 md:w-8 md:h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-gradient-to-br hover:from-purple-600 hover:to-pink-600 hover:scale-110 transition-all">
                                                <Instagram className="w-3 h-3 md:w-4 md:h-4" />
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        ))}
                        {[1, 2].map((i) => (
                            <div key={`empty-${i}`} className="aspect-[3/4] rounded-3xl bg-white/5 border border-white/10 flex items-center justify-center">
                                <div className="w-16 h-16 rounded-full bg-white/5 flex items-center justify-center">
                                    <span className="text-3xl font-bold text-white/10">?</span>
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
            </section>

            {/* 6. CTA */}
            <section className="py-24 px-6">
                <div className="max-w-3xl mx-auto">
                    <div className="relative p-12 md:p-16 rounded-3xl bg-gradient-to-br from-purple-500/10 via-zinc-900 to-blue-500/10 border border-white/10 overflow-hidden">
                        <div className="absolute top-0 right-0 w-64 h-64 bg-purple-500/20 rounded-full blur-3xl" />
                        <div className="absolute bottom-0 left-0 w-64 h-64 bg-blue-500/20 rounded-full blur-3xl" />
                        <div className="relative z-10 text-center">
                            <h2 className="text-3xl md:text-4xl font-bold mb-4">{view === 'creator' ? t.cta.title_creator : t.cta.title_studio}</h2>
                            <p className="text-white/60 mb-8 max-w-lg mx-auto">{view === 'creator' ? t.cta.subtitle_creator : t.cta.subtitle_studio}</p>
                            <Button size="lg" className="bg-white text-black hover:bg-white/90" asChild>
                                <Link href="/auth">
                                    {view === 'creator' ? t.cta.button_creator : t.cta.button_studio}
                                    <ArrowRight className="ml-2 h-4 w-4" />
                                </Link>
                            </Button>
                        </div>
                    </div>
                </div>
            </section>

            {/* 7. Footer */}
            <footer className="py-12 px-6 border-t border-white/10">
                <div className="max-w-5xl mx-auto flex flex-col md:flex-row items-center justify-between gap-6">
                    <div className="flex items-center">
                        <span className="text-lg md:text-xl font-bold tracking-tight">SWIPL</span>
                        <Image src="/swiplay-logo.png" alt="Swiplay Logo" width={20} height={20} className="inline -mx-0.5 md:w-[24px] md:h-[24px]" />
                        <span className="text-lg md:text-xl font-bold tracking-tight">Y</span>
                        <span className="text-sm text-white/40 ml-3">© 2025 {t.footer.rights}</span>
                    </div>
                    <div className="flex flex-col md:flex-row items-center gap-4 md:gap-6 text-sm text-white/40">
                        <div className="flex items-center gap-6">
                            <Link href="/legal/terms" className="hover:text-white transition-colors">{t.footer.terms}</Link>
                            <Link href="/legal/privacy" className="hover:text-white transition-colors">{t.footer.privacy}</Link>
                            <Link href="/legal/mentions" className="hover:text-white transition-colors">{t.footer.legal}</Link>
                        </div>
                        <span className="text-white/20 hidden md:inline">|</span>
                        <span className="text-xs">{locale === 'fr' ? 'Une question ?' : 'Questions?'} <a href="mailto:contact@swiplay.com" className="text-purple-400 hover:text-purple-300">contact@swiplay.com</a></span>
                    </div>
                </div>
            </footer>
        </div >
    )
}

export default LandingPage