
'use client'

import { useState, useEffect, useRef } from 'react'
import Image from 'next/image'
import { Check, X } from 'lucide-react'

export function StudioStep2Validation({ locale }: { locale: string }) {
    const [status1, setStatus1] = useState<'pending' | 'approved' | 'rejected'>('pending')
    const [status2, setStatus2] = useState<'pending' | 'approved' | 'rejected'>('pending')
    const containerRef = useRef<HTMLDivElement>(null)

    useEffect(() => {
        const observer = new IntersectionObserver(
            (entries) => {
                entries.forEach((entry) => {
                    if (entry.isIntersecting) {
                        // Reset statuses when coming into view for replayability
                        setStatus1('pending')
                        setStatus2('pending')

                        // Sequence
                        const t1 = setTimeout(() => {
                            setStatus1('approved')
                        }, 1000)

                        const t2 = setTimeout(() => {
                            setStatus2('rejected')
                        }, 2500)

                        // Clear timeouts if component unmounts or leaves view
                        return () => {
                            clearTimeout(t1)
                            clearTimeout(t2)
                        }
                    } else {
                        // Optional: Reset immediately when out of view
                        setStatus1('pending')
                        setStatus2('pending')
                    }
                })
            },
            { threshold: 0.6 } // Increased threshold to ensure user is really looking at it
        )

        if (containerRef.current) {
            observer.observe(containerRef.current)
        }

        return () => observer.disconnect()
    }, [])

    return (
        <div ref={containerRef} className="grid grid-cols-2 gap-4">
            {/* Card 1: Eyzuro (Approved) */}
            <ValidationCard
                videoSrc="/videos/eyzuro_underarock.mp4"
                creatorName="Eyzuro"
                creatorImg="/creators/eyzuro.png"
                status={status1}
                locale={locale}
            />

            {/* Card 2: Kouma (Rejected) */}
            <ValidationCard
                videoSrc="/videos/kouma.mp4"
                creatorName="Kouma"
                creatorImg="/creators/kouma.png"
                status={status2}
                locale={locale}
            />
        </div>
    )
}

function ValidationCard({
    videoSrc,
    creatorName,
    creatorImg,
    status,
    locale
}: {
    videoSrc: string
    creatorName: string
    creatorImg: string
    status: 'pending' | 'approved' | 'rejected'
    locale: string
}) {
    return (
        <div className="relative rounded-2xl bg-zinc-900/90 backdrop-blur-xl border border-white/10 shadow-2xl overflow-hidden transform hover:scale-[1.02] transition-all duration-500">
            {/* Status Badge */}
            <div className={`absolute top-3 left-3 z-10 px-3 py-1 rounded-full text-xs font-semibold shadow-lg transition-all duration-300 ${status === 'pending' ? 'bg-amber-500/90 text-white' :
                status === 'approved' ? 'bg-green-500 text-white' :
                    'bg-red-500 text-white'
                }`}>
                {status === 'pending' ? (locale === 'fr' ? 'En attente' : 'Pending') :
                    status === 'approved' ? (locale === 'fr' ? 'Validé' : 'Approved') :
                        (locale === 'fr' ? 'Refusé' : 'Rejected')}
            </div>

            {/* Video */}
            <div className="relative aspect-[9/16] bg-black">
                <video
                    className="w-full h-full object-cover opacity-80"
                    autoPlay
                    loop
                    muted
                    playsInline
                >
                    <source src={videoSrc} type="video/mp4" />
                </video>
            </div>

            {/* Overlays for Approved/Rejected state */}
            {status === 'approved' && (
                <div className="absolute inset-0 bg-green-500/20 flex items-center justify-center animate-in fade-in zoom-in duration-300">
                    <div className="w-16 h-16 rounded-full bg-green-500 flex items-center justify-center shadow-lg shadow-green-500/50">
                        <Check className="w-8 h-8 text-white stroke-[3]" />
                    </div>
                </div>
            )}

            {status === 'rejected' && (
                <div className="absolute inset-0 bg-red-500/20 flex items-center justify-center animate-in fade-in zoom-in duration-300">
                    <div className="w-16 h-16 rounded-full bg-red-500 flex items-center justify-center shadow-lg shadow-red-500/50">
                        <X className="w-8 h-8 text-white stroke-[3]" />
                    </div>
                </div>
            )}

            {/* Creator Info Footer */}
            <div className="absolute bottom-0 left-0 right-0 p-3 bg-gradient-to-t from-black/90 to-transparent">
                <div className="flex items-center gap-2">
                    <div className="w-8 h-8 rounded-full overflow-hidden ring-1 ring-white/30">
                        <Image src={creatorImg} alt={creatorName} width={32} height={32} className="w-full h-full object-cover" />
                    </div>
                    <div>
                        <p className="font-semibold text-white text-xs">{creatorName}</p>
                    </div>
                </div>
            </div>
        </div>
    )
}
