'use client'

import { ReactNode } from 'react'
import Link from 'next/link'
import { ArrowLeft, FileText, Shield, Scale, ChevronRight } from 'lucide-react'
import { Card, CardContent } from '@/components/ui'
import { LandingHeader } from '@/components/landing/LandingHeader'
import { LandingFooter } from '@/components/landing/LandingFooter'

interface LegalLayoutProps {
    title: string
    subtitle: string
    lastUpdated: string
    children: ReactNode
    icon?: 'terms' | 'privacy' | 'mentions'
}

const icons = {
    terms: Scale,
    privacy: Shield,
    mentions: FileText,
}

export function LegalLayout({ title, subtitle, lastUpdated, children, icon = 'terms' }: LegalLayoutProps) {
    const Icon = icons[icon]

    // Default to 'fr' locale for legal pages if not passed (though pages should ideally pass it)
    // For now assuming French context or generic since Legal pages structure in this repo seemed to check params locale
    // We can rely on a hardcoded prop or modify LegalLayout to accept locale.
    // Based on previous file reads, legal pages pass params.
    // However, LegalLayout usage in page.tsx didn't pass locale.
    // We'll update the interface to accept locale if needed, but for now defaulting 'fr' is safe or we can extract from URL using hook if we want to be clean.
    // Actually, let's keep it simple. The LandingHeader switches locale via router.push, so it handles itself.
    // We just need to ensure it renders.

    return (
        <div className="min-h-screen bg-black text-white">
            <LandingHeader locale="fr" />

            {/* Hero Header */}
            <div className="relative overflow-hidden border-b border-white/10 pt-20">
                <div className="absolute inset-0 bg-gradient-to-br from-violet-600/10 via-purple-600/5 to-transparent" />

                <div className="relative max-w-4xl mx-auto px-6 py-16">
                    <Link href="/" className="inline-flex items-center gap-2 text-sm text-white/60 hover:text-white transition-colors mb-8">
                        <ArrowLeft className="h-4 w-4" />
                        Retour à l'accueil
                    </Link>

                    <div className="flex items-start gap-5">
                        <div className="w-14 h-14 rounded-2xl bg-gradient-to-br from-violet-500 to-purple-600 flex items-center justify-center shadow-lg shadow-violet-500/20">
                            <Icon className="h-7 w-7 text-white" />
                        </div>
                        <div>
                            <h1 className="text-3xl md:text-4xl font-bold tracking-tight mb-2 text-white">{title}</h1>
                            <p className="text-white/60">{subtitle}</p>
                            <p className="text-sm text-white/40 mt-2">Dernière mise à jour : {lastUpdated}</p>
                        </div>
                    </div>
                </div>
            </div>

            {/* Content */}
            <div className="max-w-4xl mx-auto px-6 py-12">
                <Card className="border-white/10 bg-zinc-900/30">
                    <CardContent className="p-8 md:p-12">
                        <div className="prose prose-invert prose-lg max-w-none text-white/80">
                            {children}
                        </div>
                    </CardContent>
                </Card>

                {/* Quick Links removed as per user request */}
            </div>

            <LandingFooter locale="fr" />
        </div>
    )
}
