'use client'

import { useState } from 'react'
import { Link } from '@/i18n/routing'
import { usePathname } from 'next/navigation'
import Image from 'next/image'
import { cn } from '@/lib/utils'
import {
    LayoutDashboard, Upload, DollarSign, User, Settings,
    Megaphone, Users, CreditCard, BarChart3, FileText, ShieldAlert,
    LogOut, Menu, X, LucideIcon
} from 'lucide-react'
import { signOut } from 'next-auth/react'
import { useTranslations } from 'next-intl'

const iconMap: Record<string, LucideIcon> = {
    LayoutDashboard, Upload, DollarSign, User, Settings,
    Megaphone, Users, CreditCard, BarChart3, FileText, ShieldAlert
}

interface NavItem {
    iconName: string
    label: string
    href: string
}

interface SidebarProps {
    items: NavItem[]
    role: 'creator' | 'studio' | 'admin'
    user?: { name?: string | null; email?: string | null; image?: string | null }
}

import { useAuth } from '@/lib/auth-store'

export function Sidebar({ items, role, user }: SidebarProps) {
    const t = useTranslations('Dashboard.Sidebar')
    const pathname = usePathname()
    const [isOpen, setIsOpen] = useState(false)

    // Subscribe to auth store updates, fallback to props
    const storeUser = useAuth((state) => state.user)
    const currentUser = storeUser || user

    return (
        <>
            {/* Mobile toggle */}
            <button
                onClick={() => setIsOpen(!isOpen)}
                className="fixed top-4 left-4 z-50 p-2 rounded-lg bg-card border border-border md:hidden"
            >
                {isOpen ? <X className="w-5 h-5" /> : <Menu className="w-5 h-5" />}
            </button>

            {/* Overlay */}
            {isOpen && (
                <div
                    className="fixed inset-0 bg-black/60 z-40 md:hidden"
                    onClick={() => setIsOpen(false)}
                />
            )}

            {/* Sidebar */}
            <aside className={cn(
                "fixed left-0 top-0 bottom-0 w-64 bg-card border-r border-border z-50 flex flex-col transition-transform duration-300",
                "md:translate-x-0",
                isOpen ? "translate-x-0" : "-translate-x-full"
            )}>
                {/* Logo */}
                <div className="h-16 flex items-center px-6 border-b border-border">
                    <Link href={`/${role}`} className="flex items-center gap-3">
                        <Image
                            src="/swiplay-logo.png"
                            alt="Swiplay"
                            width={28}
                            height={28}
                            className="opacity-90"
                        />
                        <span className="text-lg font-semibold tracking-tight">Swiplay</span>
                    </Link>
                </div>

                {/* Navigation */}
                <nav className="flex-1 py-4 px-3 overflow-y-auto">
                    <ul className="space-y-1">
                        {items.map((item) => {
                            const Icon = iconMap[item.iconName] || LayoutDashboard
                            const isActive = pathname.includes(item.href)

                            return (
                                <li key={item.href}>
                                    <Link
                                        href={item.href}
                                        onClick={() => setIsOpen(false)}
                                        className={cn(
                                            "flex items-center gap-3 px-3 py-2.5 rounded-lg text-sm font-medium transition-colors",
                                            isActive
                                                ? "bg-white text-black"
                                                : "text-muted-foreground hover:text-foreground hover:bg-secondary"
                                        )}
                                    >
                                        <Icon className="w-[18px] h-[18px]" />
                                        {item.label}
                                    </Link>
                                </li>
                            )
                        })}
                    </ul>
                </nav>

                {/* User & Logout */}
                <div className="p-4 border-t border-border">
                    <div className="flex items-center gap-3 mb-4">
                        <div className="w-9 h-9 rounded-full bg-secondary flex items-center justify-center text-sm font-medium overflow-hidden">
                            {currentUser?.image ? (
                                <Image
                                    src={currentUser.image}
                                    alt={currentUser?.name || 'Avatar'}
                                    width={36}
                                    height={36}
                                    className="object-cover w-full h-full"
                                />
                            ) : (
                                currentUser?.name?.[0]?.toUpperCase() || currentUser?.email?.[0]?.toUpperCase() || 'U'
                            )}
                        </div>
                        <div className="flex-1 min-w-0">
                            <p className="text-sm font-medium truncate">{currentUser?.name || currentUser?.email}</p>
                            <p className="text-xs text-muted-foreground capitalize">{role}</p>
                        </div>
                    </div>
                    <button
                        onClick={() => signOut({ callbackUrl: 'https://swiplay.com' })}
                        className="flex items-center gap-2 w-full px-3 py-2 rounded-lg text-sm text-muted-foreground hover:text-foreground hover:bg-secondary transition-colors"
                    >
                        <LogOut className="w-4 h-4" />
                        {t('logout')}
                    </button>
                </div>
            </aside>
        </>
    )
}
