import { NextResponse } from 'next/server'
import { ZodError } from 'zod'

export interface ApiError {
  message: string
  code?: string
  details?: any
}

export function handleApiError(error: unknown): NextResponse<ApiError> {
  console.error('API Error:', error)

  if (error instanceof ZodError) {
    return NextResponse.json(
      {
        message: 'Validation failed',
        code: 'VALIDATION_ERROR',
        details: error.errors
      },
      { status: 400 }
    )
  }

  if (error instanceof Error) {
    // Database errors
    if (error.message.includes('duplicate key')) {
      return NextResponse.json(
        {
          message: 'Resource already exists',
          code: 'DUPLICATE_ERROR'
        },
        { status: 409 }
      )
    }

    // JWT errors
    if (error.message.includes('jwt')) {
      return NextResponse.json(
        {
          message: 'Authentication error',
          code: 'AUTH_ERROR'
        },
        { status: 401 }
      )
    }

    // Generic error
    return NextResponse.json(
      {
        message: 'Internal server error',
        code: 'INTERNAL_ERROR'
      },
      { status: 500 }
    )
  }

  return NextResponse.json(
    {
      message: 'Unknown error occurred',
      code: 'UNKNOWN_ERROR'
    },
    { status: 500 }
  )
}

export function createApiResponse<T>(
  data: T,
  status: number = 200
): NextResponse<{ data: T }> {
  return NextResponse.json({ data }, { status })
}

export function createErrorResponse(
  message: string,
  code: string = 'ERROR',
  status: number = 400
): NextResponse<ApiError> {
  return NextResponse.json(
    { message, code },
    { status }
  )
}