const API_URL = process.env.NEXT_PUBLIC_API_URL || '/api'

export interface ApiResponse<T> {
  data?: T
  error?: string
}

async function getAuthToken(): Promise<string | null> {
  // Get from localStorage (Zustand persist)
  const authStorage = localStorage.getItem('auth-storage')
  if (!authStorage) return null

  const auth = JSON.parse(authStorage).state
  return auth.accessToken
}

async function refreshTokenIfNeeded(): Promise<string | null> {
  const authStorage = localStorage.getItem('auth-storage')
  if (!authStorage) return null

  const auth = JSON.parse(authStorage).state
  if (!auth.refreshToken) return null

  try {
    const response = await fetch(`${API_URL}/auth/refresh`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ refreshToken: auth.refreshToken })
    })

    if (!response.ok) return null

    const data = await response.json()

    // Update localStorage
    const updatedAuth = { ...auth, accessToken: data.accessToken, refreshToken: data.refreshToken }
    localStorage.setItem('auth-storage', JSON.stringify({ state: updatedAuth, version: 0 }))

    return data.accessToken
  } catch (error) {
    console.error('Token refresh failed:', error)
    return null
  }
}

async function fetchApi<T>(
  endpoint: string,
  options: RequestInit = {}
): Promise<ApiResponse<T>> {
  try {
    let token = await getAuthToken()

    const makeRequest = (authToken?: string) => {
      const headers: Record<string, string> = {
        'Content-Type': 'application/json',
        ...(authToken ? { Authorization: `Bearer ${authToken}` } : {}),
        ...((options.headers as Record<string, string>) || {}),
      }

      return fetch(`${API_URL}${endpoint}`, {
        ...options,
        headers,
      })
    }

    let response = await makeRequest(token)

    // If unauthorized, try to refresh token
    if (response.status === 401) {
      token = await refreshTokenIfNeeded()
      if (token) {
        response = await makeRequest(token)
      }
    }

    const data = await response.json()

    if (!response.ok) {
      return { error: data.error || 'An error occurred' }
    }

    return { data }
  } catch (error) {
    return { error: 'Network error' }
  }
}

// Auth API
export const authApi = {
  register: (data: {
    email: string
    password: string
    name: string
    role: 'studio' | 'creator'
  }) =>
    fetchApi<{ user: any; accessToken: string; refreshToken: string }>('/auth/register', {
      method: 'POST',
      body: JSON.stringify(data),
    }),

  login: (data: { email: string; password: string }) =>
    fetchApi<{ user: any; accessToken: string; refreshToken: string }>('/auth/login', {
      method: 'POST',
      body: JSON.stringify(data),
    }),

  me: () => fetchApi<any>('/auth/me'),
}

// Campaigns API
export const campaignsApi = {
  list: () => fetchApi<any[]>('/campaigns'),

  get: (id: string) => fetchApi<any>(`/campaigns/${id}`),

  create: (data: {
    title: string
    description: string
    budget: number
    pricePerView: number
    startDate?: string
    endDate?: string
    creatorCap?: number
    coverImage?: string
    icon?: string
    images?: string[]
    videos?: string[]
    tags?: string[]
    gameKeys?: { platform: string; keys: string }[]
    assets?: { name: string; type: string; url: string; size?: number }[]
  }) =>
    fetchApi<any>('/campaigns', {
      method: 'POST',
      body: JSON.stringify(data),
    }),

  update: (id: string, data: any) =>
    fetchApi<any>(`/campaigns/${id}`, {
      method: 'PATCH',
      body: JSON.stringify(data),
    }),

  delete: (id: string) =>
    fetchApi<any>(`/campaigns/${id}`, {
      method: 'DELETE',
    }),
}

// Submissions API
export const submissionsApi = {
  list: () => fetchApi<any[]>('/submissions'),

  get: (id: string) => fetchApi<any>(`/submissions/${id}`),

  create: (data: {
    campaignId: string
    videoUrl: string
    platform: 'tiktok' | 'youtube' | 'instagram'
  }) =>
    fetchApi<any>('/submissions', {
      method: 'POST',
      body: JSON.stringify(data),
    }),

  approve: (id: string) =>
    fetchApi<any>(`/submissions/${id}/approve`, {
      method: 'PATCH',
    }),

  reject: (id: string) =>
    fetchApi<any>(`/submissions/${id}/reject`, {
      method: 'PATCH',
    }),
}
