import { pgTable, uuid, varchar, text, decimal, timestamp, pgEnum, integer, uniqueIndex, bigint, index } from 'drizzle-orm/pg-core'
import { relations } from 'drizzle-orm'

// Enums
export const userRoleEnum = pgEnum('user_role', ['studio', 'creator', 'admin'])
export const campaignStatusEnum = pgEnum('campaign_status', ['draft', 'pending_approval', 'active', 'paused', 'completed', 'rejected'])
export const submissionStatusEnum = pgEnum('submission_status', ['pending', 'approved', 'rejected'])
export const platformEnum = pgEnum('platform', ['tiktok', 'youtube', 'instagram'])
export const paymentStatusEnum = pgEnum('payment_status', ['pending', 'processing', 'paid', 'failed'])
export const verificationStatusEnum = pgEnum('verification_status', ['pending', 'approved', 'rejected'])
export const applicationStatusEnum = pgEnum('application_status', ['pending', 'approved', 'rejected'])
export const payoutStatusEnum = pgEnum('payout_status', ['pending', 'paid', 'rejected'])

// Users Table (NextAuth + App Specific)
export const users = pgTable('users', {
  id: uuid('id').defaultRandom().primaryKey(),
  name: varchar('name', { length: 255 }),
  email: varchar('email', { length: 255 }).notNull(),
  emailVerified: timestamp('emailVerified', { mode: 'date' }),
  image: varchar('image', { length: 500 }), // Replaces 'avatar'
  role: userRoleEnum('role').default('creator').notNull(),
  company: varchar('company', { length: 255 }), // For studios
  isApproved: integer('is_approved').default(0).notNull(), // 0 = pending, 1 = approved

  // New Fields
  discordHandle: varchar('discord_handle', { length: 255 }),
  bio: text('bio'), // Creator bio visible to studios
  privacyPolicyAccepted: integer('privacy_policy_accepted').default(0), // 0 = no, 1 = yes
  termsAccepted: integer('terms_accepted').default(0), // 0 = no, 1 = yes
  balance: decimal('balance', { precision: 10, scale: 2 }).default('0').notNull(),

  // Social Media OAuth Tokens (kept for app logic, though NextAuth handles accounts separately)
  tiktokAccessToken: text('tiktok_access_token'),
  tiktokRefreshToken: text('tiktok_refresh_token'),
  tiktokUserId: varchar('tiktok_user_id', { length: 255 }),
  youtubeAccessToken: text('youtube_access_token'),
  youtubeRefreshToken: text('youtube_refresh_token'),
  youtubeChannelId: varchar('youtube_channel_id', { length: 255 }),
  instagramAccessToken: text('instagram_access_token'),
  instagramRefreshToken: text('instagram_refresh_token'),
  instagramUserId: varchar('instagram_user_id', { length: 255 }),

  createdAt: timestamp('created_at').defaultNow().notNull(),
  updatedAt: timestamp('updated_at').defaultNow().notNull(),

  // Payment Info
  currency: varchar('currency', { length: 3 }).default('EUR'), // 'EUR' | 'USD'
  iban: text('iban'), // For EUR
  bic: text('bic'), // For EUR & USD
  accountNumber: text('account_number'), // For USD
  routingNumber: text('routing_number'), // For USD
  bankName: text('bank_name'), // Optional
  bankAddress: text('bank_address'), // Optional
  onboardingComplete: integer('onboarding_complete').default(0).notNull(), // 0 = false, 1 = true
  preferredLocale: varchar('preferred_locale', { length: 5 }).default('fr'), // 'fr' | 'en'
}, (table) => ({
  emailRoleUnique: uniqueIndex('users_email_role_unique').on(table.email, table.role),
}))

// NextAuth Accounts Table
export const accounts = pgTable(
  "account",
  {
    userId: uuid("userId")
      .notNull()
      .references(() => users.id, { onDelete: "cascade" }),
    type: text("type").notNull(),
    provider: text("provider").notNull(),
    providerAccountId: text("providerAccountId").notNull(),
    refresh_token: text("refresh_token"),
    access_token: text("access_token"),
    expires_at: integer("expires_at"),
    token_type: text("token_type"),
    scope: text("scope"),
    id_token: text("id_token"),
    session_state: text("session_state"),
  },
  (account) => ({
    compoundKey: uniqueIndex("account_provider_providerAccountId_unique").on(
      account.provider,
      account.providerAccountId
    ),
  })
)

// NextAuth Sessions Table
export const sessions = pgTable("session", {
  sessionToken: text("sessionToken").primaryKey(),
  userId: uuid("userId")
    .notNull()
    .references(() => users.id, { onDelete: "cascade" }),
  expires: timestamp("expires", { mode: "date" }).notNull(),
})

// NextAuth Verification Tokens Table
export const verificationTokens = pgTable(
  "verificationToken",
  {
    identifier: text("identifier").notNull(),
    token: text("token").notNull(),
    expires: timestamp("expires", { mode: "date" }).notNull(),
  },
  (vt) => ({
    compoundKey: uniqueIndex("verificationToken_identifier_token_unique").on(
      vt.identifier,
      vt.token
    ),
  })
)

// Login Codes Table (OTP / Magic Link) - Keeping for custom auth if needed, or can be deprecated
export const loginCodes = pgTable('login_codes', {
  id: uuid('id').defaultRandom().primaryKey(),
  email: varchar('email', { length: 255 }).notNull(),
  code: varchar('code', { length: 6 }).notNull(),
  role: userRoleEnum('role').notNull(),
  expiresAt: timestamp('expires_at').notNull(),
  usedAt: timestamp('used_at'),
  createdAt: timestamp('created_at').defaultNow().notNull(),
})

// Refresh Tokens Table - Keeping for now, might be redundant with NextAuth
export const refreshTokens = pgTable('refresh_tokens', {
  id: uuid('id').defaultRandom().primaryKey(),
  userId: uuid('user_id').references(() => users.id, { onDelete: 'cascade' }).notNull(),
  token: varchar('token', { length: 500 }).notNull().unique(),
  expiresAt: timestamp('expires_at').notNull(),
  createdAt: timestamp('created_at').defaultNow().notNull(),
})

// Campaigns Table
export const campaigns = pgTable('campaigns', {
  id: uuid('id').defaultRandom().primaryKey(),
  studioId: uuid('studio_id').references(() => users.id, { onDelete: 'cascade' }).notNull(),
  title: varchar('title', { length: 255 }).notNull(),
  description: text('description').notNull(),
  budget: decimal('budget', { precision: 10, scale: 2 }).notNull(),
  pricePerView: decimal('price_per_view', { precision: 10, scale: 4 }).notNull(),
  creatorCap: decimal('creator_cap', { precision: 10, scale: 2 }), // Max earnings per creator
  status: campaignStatusEnum('status').default('draft').notNull(),

  // Visuals
  coverImage: varchar('cover_image', { length: 500 }), // 16:9 Banner
  backgroundImage: varchar('background_image', { length: 500 }),
  icon: varchar('icon', { length: 500 }),
  images: text('images').array(), // Array of image URLs (1-10)
  videos: text('videos').array(), // Array of video URLs (1-3)
  tags: text('tags').array(), // Array of game tags

  totalStorageUsed: bigint('total_storage_used', { mode: 'number' }).default(0).notNull(),
  storageLimit: bigint('storage_limit', { mode: 'number' }).default(5368709120).notNull(), // 5GB default
  startDate: timestamp('start_date'),
  endDate: timestamp('end_date'),
  paymentIntentId: varchar('payment_intent_id', { length: 255 }), // For refunds
  createdAt: timestamp('created_at').defaultNow().notNull(),
  updatedAt: timestamp('updated_at').defaultNow().notNull(),
}, (table) => ({
  studioIdIdx: index('campaigns_studio_id_idx').on(table.studioId),
  statusIdx: index('campaigns_status_idx').on(table.status),
}))

// Creator Socials Table
export const creatorSocials = pgTable('creator_socials', {
  id: uuid('id').defaultRandom().primaryKey(),
  creatorId: uuid('creator_id').references(() => users.id, { onDelete: 'cascade' }).notNull(),
  platform: platformEnum('platform').notNull(),
  handle: varchar('handle', { length: 100 }).notNull(),
  followers: integer('followers').default(0),
  avgViews: integer('avg_views').default(0),
  profileUrl: varchar('profile_url', { length: 500 }),
  screenshotUrl: varchar('screenshot_url', { length: 500 }),
  verificationMethod: varchar('verification_method', { length: 50 }).default('oauth'), // 'oauth' or 'screenshot'
  rejectionReason: text('rejection_reason'),
  verificationStatus: verificationStatusEnum('verification_status').default('pending').notNull(),
  createdAt: timestamp('created_at').defaultNow().notNull(),
  updatedAt: timestamp('updated_at').defaultNow().notNull(),
}, (table) => ({
  creatorIdPlatformUnique: uniqueIndex('creator_socials_creator_id_platform_unique').on(table.creatorId, table.platform),
  statusIdx: index('creator_socials_status_idx').on(table.verificationStatus),
}))

export const creatorSocialsRelations = relations(creatorSocials, ({ one }) => ({
  creator: one(users, {
    fields: [creatorSocials.creatorId],
    references: [users.id],
  }),
}))

// Game Keys Table
export const gameKeys = pgTable('game_keys', {
  id: uuid('id').defaultRandom().primaryKey(),
  campaignId: uuid('campaign_id').references(() => campaigns.id, { onDelete: 'cascade' }).notNull(),
  key: varchar('key', { length: 255 }).notNull(),
  platform: varchar('platform', { length: 50 }).default('PC').notNull(), // New platform field
  isClaimed: integer('is_claimed').default(0).notNull(), // 0 = false, 1 = true
  claimedBy: uuid('claimed_by').references(() => users.id, { onDelete: 'set null' }),
  claimedAt: timestamp('claimed_at'),
  createdAt: timestamp('created_at').defaultNow().notNull(),
}, (table) => ({
  campaignIdIdx: index('game_keys_campaign_id_idx').on(table.campaignId),
  isClaimedIdx: index('game_keys_is_claimed_idx').on(table.isClaimed),
}))

// Campaign Resources Table (Drive, Discord, YouTube links)
export const campaignResources = pgTable('campaign_resources', {
  id: uuid('id').defaultRandom().primaryKey(),
  campaignId: uuid('campaign_id').references(() => campaigns.id, { onDelete: 'cascade' }).notNull(),
  type: varchar('type', { length: 50 }).notNull(), // 'drive', 'discord', 'youtube', 'link'
  title: varchar('title', { length: 255 }).notNull(),
  url: varchar('url', { length: 500 }).notNull(),
  description: text('description'),
  createdAt: timestamp('created_at').defaultNow().notNull(),
}, (table) => ({
  campaignIdIdx: index('campaign_resources_campaign_id_idx').on(table.campaignId),
}))

// Submissions Table
export const submissions = pgTable('submissions', {
  id: uuid('id').defaultRandom().primaryKey(),
  campaignId: uuid('campaign_id').references(() => campaigns.id, { onDelete: 'cascade' }).notNull(),
  creatorId: uuid('creator_id').references(() => users.id, { onDelete: 'cascade' }).notNull(),
  videoUrl: varchar('video_url', { length: 500 }).notNull(),
  platform: platformEnum('platform').notNull(),
  status: submissionStatusEnum('status').default('pending').notNull(),
  viewsCount: integer('views_count').default(0).notNull(),
  revenue: decimal('revenue', { precision: 10, scale: 2 }).default('0').notNull(),
  likes: integer('likes').default(0).notNull(),
  engagementRate: decimal('engagement_rate', { precision: 5, scale: 2 }).default('0').notNull(),
  lastUpdated: timestamp('last_updated'),
  submittedAt: timestamp('submitted_at').defaultNow().notNull(),
  reviewedAt: timestamp('reviewed_at'),
  invoiceId: uuid('invoice_id'), // Link to the generated invoice
  legalCertified: integer('legal_certified').default(0).notNull(), // 1 = creator certified compliance
  createdAt: timestamp('created_at').defaultNow().notNull(),
  updatedAt: timestamp('updated_at').defaultNow().notNull(),
}, (table) => ({
  campaignIdIdx: index('submissions_campaign_id_idx').on(table.campaignId),
  creatorIdIdx: index('submissions_creator_id_idx').on(table.creatorId),
  statusIdx: index('submissions_status_idx').on(table.status),
  invoiceIdIdx: index('submissions_invoice_id_idx').on(table.invoiceId),
}))

// Submission Stats History Table
export const submissionStatsHistory = pgTable('submission_stats_history', {
  id: uuid('id').defaultRandom().primaryKey(),
  submissionId: uuid('submission_id').references(() => submissions.id, { onDelete: 'cascade' }).notNull(),
  views: integer('views').default(0).notNull(),
  likes: integer('likes').default(0).notNull(),
  engagementRate: decimal('engagement_rate', { precision: 5, scale: 2 }).default('0').notNull(),
  recordedAt: timestamp('recorded_at').defaultNow().notNull(),
}, (table) => ({
  submissionIdIdx: index('submission_stats_history_submission_id_idx').on(table.submissionId),
  recordedAtIdx: index('submission_stats_history_recorded_at_idx').on(table.recordedAt),
}))

export const submissionStatsHistoryRelations = relations(submissionStatsHistory, ({ one }) => ({
  submission: one(submissions, {
    fields: [submissionStatsHistory.submissionId],
    references: [submissions.id],
  }),
}))

// Invoices Table
export const invoices = pgTable('invoices', {
  id: uuid('id').defaultRandom().primaryKey(),
  creatorId: uuid('creator_id').references(() => users.id, { onDelete: 'cascade' }), // Optional now
  studioId: uuid('studio_id').references(() => users.id, { onDelete: 'cascade' }).notNull(),
  campaignId: uuid('campaign_id').references(() => campaigns.id, { onDelete: 'set null' }), // Link to campaign
  amount: decimal('amount', { precision: 10, scale: 2 }).notNull(),
  currency: varchar('currency', { length: 3 }).default('EUR').notNull(),
  pdfUrl: varchar('pdf_url', { length: 500 }).notNull(),
  invoiceNumber: varchar('invoice_number', { length: 100 }).notNull(),
  periodStart: timestamp('period_start'),
  periodEnd: timestamp('period_end'),
  status: varchar('status', { length: 20 }).default('pending').notNull(), // 'pending', 'pending_verification', 'paid', 'failed'
  paymentReference: varchar('payment_reference', { length: 255 }),
  paidAt: timestamp('paid_at'),
  createdAt: timestamp('created_at').defaultNow().notNull(),
}, (table) => ({
  creatorIdIdx: index('invoices_creator_id_idx').on(table.creatorId),
  studioIdIdx: index('invoices_studio_id_idx').on(table.studioId),
  campaignIdIdx: index('invoices_campaign_id_idx').on(table.campaignId),
}))

// Campaign Applications Table
export const campaignApplications = pgTable('campaign_applications', {
  id: uuid('id').defaultRandom().primaryKey(),
  campaignId: uuid('campaign_id').references(() => campaigns.id, { onDelete: 'cascade' }).notNull(),
  creatorId: uuid('creator_id').references(() => users.id, { onDelete: 'cascade' }).notNull(),
  status: applicationStatusEnum('status').default('pending').notNull(),
  createdAt: timestamp('created_at').defaultNow().notNull(),
  updatedAt: timestamp('updated_at').defaultNow().notNull(),
}, (table) => ({
  campaignIdIdx: index('campaign_applications_campaign_id_idx').on(table.campaignId),
  creatorIdIdx: index('campaign_applications_creator_id_idx').on(table.creatorId),
  statusIdx: index('campaign_applications_status_idx').on(table.status),
  uniqueApplication: uniqueIndex('campaign_applications_unique').on(table.campaignId, table.creatorId),
}))

// Payout Requests Table
export const payoutRequests = pgTable('payout_requests', {
  id: uuid('id').defaultRandom().primaryKey(),
  creatorId: uuid('creator_id').references(() => users.id, { onDelete: 'cascade' }).notNull(),
  amount: decimal('amount', { precision: 10, scale: 2 }).notNull(),
  status: payoutStatusEnum('status').default('pending').notNull(),
  processedAt: timestamp('processed_at'),
  createdAt: timestamp('created_at').defaultNow().notNull(),
}, (table) => ({
  creatorIdIdx: index('payout_requests_creator_id_idx').on(table.creatorId),
  statusIdx: index('payout_requests_status_idx').on(table.status),
}))

// Payments Table (Future phase)
export const payments = pgTable('payments', {
  id: uuid('id').defaultRandom().primaryKey(),
  creatorId: uuid('creator_id').references(() => users.id, { onDelete: 'cascade' }).notNull(),
  submissionId: uuid('submission_id').references(() => submissions.id, { onDelete: 'cascade' }).notNull(),
  amount: decimal('amount', { precision: 10, scale: 2 }).notNull(),
  status: paymentStatusEnum('status').default('pending').notNull(),
  paidAt: timestamp('paid_at'),
  createdAt: timestamp('created_at').defaultNow().notNull(),
})

export const submissionsRelations = relations(submissions, ({ one, many }) => ({
  campaign: one(campaigns, {
    fields: [submissions.campaignId],
    references: [campaigns.id],
  }),
  creator: one(users, {
    fields: [submissions.creatorId],
    references: [users.id],
  }),
  invoice: one(invoices, {
    fields: [submissions.invoiceId],
    references: [invoices.id],
  }),
  payments: many(payments),
}))

export const invoicesRelations = relations(invoices, ({ one, many }) => ({
  creator: one(users, {
    fields: [invoices.creatorId],
    references: [users.id],
    relationName: 'creatorInvoices',
  }),
  studio: one(users, {
    fields: [invoices.studioId],
    references: [users.id],
    relationName: 'studioInvoices',
  }),
  campaign: one(campaigns, {
    fields: [invoices.campaignId],
    references: [campaigns.id],
  }),
  submissions: many(submissions),
}))

export const paymentsRelations = relations(payments, ({ one }) => ({
  creator: one(users, {
    fields: [payments.creatorId],
    references: [users.id],
  }),
  submission: one(submissions, {
    fields: [payments.submissionId],
    references: [submissions.id],
  }),
}))

export const refreshTokensRelations = relations(refreshTokens, ({ one }) => ({
  user: one(users, {
    fields: [refreshTokens.userId],
    references: [users.id],
  }),
}))

// Creator Profiles Table (Social Media Handles)
export const creatorProfiles = pgTable('creator_profiles', {
  id: uuid('id').defaultRandom().primaryKey(),
  creatorId: uuid('creator_id').references(() => users.id, { onDelete: 'cascade' }).notNull().unique(),
  tiktok: varchar('tiktok', { length: 100 }),
  youtube: varchar('youtube', { length: 100 }),
  instagram: varchar('instagram', { length: 100 }),
  twitch: varchar('twitch', { length: 100 }),
  followers: integer('followers').default(0),
  avgViews: integer('avg_views').default(0),
  bankName: varchar('bank_name', { length: 255 }),
  iban: varchar('iban', { length: 255 }),
  bic: varchar('bic', { length: 255 }),
  paypalEmail: varchar('paypal_email', { length: 255 }),
  bio: text('bio'), // New bio field
  // Billing Info
  firstName: varchar('first_name', { length: 255 }),
  lastName: varchar('last_name', { length: 255 }),
  legalName: varchar('legal_name', { length: 255 }), // Deprecated, kept for legacy
  address: varchar('address', { length: 500 }),
  city: varchar('city', { length: 255 }),
  zip: varchar('zip', { length: 20 }),
  country: varchar('country', { length: 255 }),
  createdAt: timestamp('created_at').defaultNow().notNull(),
  updatedAt: timestamp('updated_at').defaultNow().notNull(),
})

// User Settings Table
export const userSettings = pgTable('user_settings', {
  id: uuid('id').defaultRandom().primaryKey(),
  userId: uuid('user_id').references(() => users.id, { onDelete: 'cascade' }).notNull().unique(),
  // Notifications
  emailNotifications: integer('email_notifications').default(1).notNull(), // 0 = false, 1 = true
  newCampaignsNotifications: integer('new_campaigns_notifications').default(1).notNull(),
  submissionStatusNotifications: integer('submission_status_notifications').default(1).notNull(),
  paymentNotifications: integer('payment_notifications').default(1).notNull(),
  marketingEmails: integer('marketing_emails').default(0).notNull(),
  // Privacy
  profileVisible: integer('profile_visible').default(1).notNull(),
  statsVisible: integer('stats_visible').default(0).notNull(),
  createdAt: timestamp('created_at').defaultNow().notNull(),
  updatedAt: timestamp('updated_at').defaultNow().notNull(),
})

// Campaign Assets Table (Google Drive metadata for campaign assets)
export const campaignAssets = pgTable('campaign_assets', {
  id: uuid('id').defaultRandom().primaryKey(),
  campaignId: uuid('campaign_id').references(() => campaigns.id, { onDelete: 'cascade' }).notNull(),
  fileId: varchar('file_id', { length: 255 }).notNull(), // Google Drive file ID
  fileName: varchar('file_name', { length: 500 }).notNull(),
  originalName: varchar('original_name', { length: 500 }),
  mimeType: varchar('mime_type', { length: 255 }),
  fileType: varchar('file_type', { length: 50 }), // 'video', 'image', 'gamekey', 'other'
  driveUrl: varchar('drive_url', { length: 500 }), // Google Drive public URL
  size: bigint('size', { mode: 'number' }),
  uploadedAt: timestamp('uploaded_at').defaultNow().notNull(),
})

// Notifications Table
export const notifications = pgTable('notifications', {
  id: uuid('id').defaultRandom().primaryKey(),
  userId: uuid('user_id').references(() => users.id, { onDelete: 'cascade' }).notNull(),
  type: varchar('type', { length: 50 }).notNull(), // 'info', 'success', 'warning', 'error'
  title: varchar('title', { length: 255 }).notNull(),
  message: text('message').notNull(),
  read: integer('read').default(0).notNull(), // 0 = unread, 1 = read
  link: varchar('link', { length: 500 }),
  createdAt: timestamp('created_at').defaultNow().notNull(),
}, (table) => ({
  userIdIdx: uniqueIndex('notifications_user_id_idx').on(table.userId, table.createdAt),
}))

// Global Settings Table (Singleton)
export const globalSettings = pgTable('global_settings', {
  id: uuid('id').defaultRandom().primaryKey(),
  platformFee: integer('platform_fee').default(10).notNull(), // Percentage
  minPayout: integer('min_payout').default(50).notNull(), // EUR
  autoApproveCreators: integer('auto_approve_creators').default(0).notNull(), // 0 = false, 1 = true
  requireStudioApproval: integer('require_studio_approval').default(1).notNull(), // 0 = false, 1 = true
  emailNotifications: integer('email_notifications').default(1).notNull(), // 0 = false, 1 = true
  weeklyReports: integer('weekly_reports').default(1).notNull(), // 0 = false, 1 = true
  maintenanceMode: integer('maintenance_mode').default(0).notNull(), // 0 = false, 1 = true
  autoMonthlyInvoicing: integer('auto_monthly_invoicing').default(0).notNull(), // 0 = false, 1 = true
  discordWebhookUrl: varchar('discord_webhook_url', { length: 500 }), // Optional override

  // Bank Details EUR
  bankEurBeneficiary: varchar('bank_eur_beneficiary', { length: 255 }),
  bankEurIban: varchar('bank_eur_iban', { length: 50 }),
  bankEurBic: varchar('bank_eur_bic', { length: 20 }),
  bankEurBankName: varchar('bank_eur_bank_name', { length: 100 }),
  bankEurBankAddress: varchar('bank_eur_bank_address', { length: 255 }),

  // Bank Details USD
  bankUsdBeneficiary: varchar('bank_usd_beneficiary', { length: 255 }),
  bankUsdAccountNumber: varchar('bank_usd_account_number', { length: 50 }),
  bankUsdRoutingNumber: varchar('bank_usd_routing_number', { length: 20 }),
  bankUsdBic: varchar('bank_usd_bic', { length: 20 }),
  bankUsdAccountType: varchar('bank_usd_account_type', { length: 20 }),
  bankUsdBankName: varchar('bank_usd_bank_name', { length: 100 }),
  bankUsdBankAddress: varchar('bank_usd_bank_address', { length: 255 }),

  updatedAt: timestamp('updated_at').defaultNow().notNull(),
})

export const creatorProfilesRelations = relations(creatorProfiles, ({ one }) => ({
  creator: one(users, {
    fields: [creatorProfiles.creatorId],
    references: [users.id],
  }),
}))


export const userSettingsRelations = relations(userSettings, ({ one }) => ({
  user: one(users, {
    fields: [userSettings.userId],
    references: [users.id],
  }),
}))

export const campaignsRelations = relations(campaigns, ({ one, many }) => ({
  studio: one(users, {
    fields: [campaigns.studioId],
    references: [users.id],
  }),
  submissions: many(submissions),
  gameKeys: many(gameKeys),
  assets: many(campaignAssets),
  applications: many(campaignApplications),
}))

export const campaignApplicationsRelations = relations(campaignApplications, ({ one }) => ({
  campaign: one(campaigns, {
    fields: [campaignApplications.campaignId],
    references: [campaigns.id],
  }),
  creator: one(users, {
    fields: [campaignApplications.creatorId],
    references: [users.id],
  }),
}))

export const payoutRequestsRelations = relations(payoutRequests, ({ one }) => ({
  creator: one(users, {
    fields: [payoutRequests.creatorId],
    references: [users.id],
  }),
}))

export const usersRelations = relations(users, ({ one, many }) => ({
  campaigns: many(campaigns), // As studio
  submissions: many(submissions), // As creator
  applications: many(campaignApplications), // As creator
  payoutRequests: many(payoutRequests), // As creator
  profile: one(creatorProfiles, {
    fields: [users.id],
    references: [creatorProfiles.creatorId],
  }),
  settings: one(userSettings, {
    fields: [users.id],
    references: [userSettings.userId],
  }),
  socials: many(creatorSocials),
  creatorInvoices: many(invoices, { relationName: 'creatorInvoices' }),
  studioInvoices: many(invoices, { relationName: 'studioInvoices' }),
}))

export const gameKeysRelations = relations(gameKeys, ({ one }) => ({
  campaign: one(campaigns, {
    fields: [gameKeys.campaignId],
    references: [campaigns.id],
  }),
  claimedBy: one(users, {
    fields: [gameKeys.claimedBy],
    references: [users.id],
  }),
}))

export const campaignAssetsRelations = relations(campaignAssets, ({ one }) => ({
  campaign: one(campaigns, {
    fields: [campaignAssets.campaignId],
    references: [campaigns.id],
  }),
}))

// TypeScript Types
export type User = typeof users.$inferSelect
export type NewUser = typeof users.$inferInsert

export type Campaign = typeof campaigns.$inferSelect
export type NewCampaign = typeof campaigns.$inferInsert

export type GameKey = typeof gameKeys.$inferSelect
export type NewGameKey = typeof gameKeys.$inferInsert

export type Submission = typeof submissions.$inferSelect
export type NewSubmission = typeof submissions.$inferInsert

export type SubmissionStatsHistory = typeof submissionStatsHistory.$inferSelect
export type NewSubmissionStatsHistory = typeof submissionStatsHistory.$inferInsert

export type CampaignApplication = typeof campaignApplications.$inferSelect
export type NewCampaignApplication = typeof campaignApplications.$inferInsert

export type PayoutRequest = typeof payoutRequests.$inferSelect
export type NewPayoutRequest = typeof payoutRequests.$inferInsert

export type Payment = typeof payments.$inferSelect
export type NewPayment = typeof payments.$inferInsert

export type RefreshToken = typeof refreshTokens.$inferSelect
export type NewRefreshToken = typeof refreshTokens.$inferInsert

export type CreatorProfile = typeof creatorProfiles.$inferSelect
export type NewCreatorProfile = typeof creatorProfiles.$inferInsert

export type CreatorSocial = typeof creatorSocials.$inferSelect
export type NewCreatorSocial = typeof creatorSocials.$inferInsert

export type UserSetting = typeof userSettings.$inferSelect
export type NewUserSetting = typeof userSettings.$inferInsert

export type Notification = typeof notifications.$inferSelect
export type NewNotification = typeof notifications.$inferInsert

export type GlobalSettings = typeof globalSettings.$inferSelect
export type NewGlobalSettings = typeof globalSettings.$inferInsert

// Rate Limits Table (Persistent)
export const rateLimits = pgTable('rate_limits', {
  key: varchar('key', { length: 255 }).primaryKey(), // IP or Token
  count: integer('count').default(0).notNull(),
  expiresAt: bigint('expires_at', { mode: 'number' }).notNull(), // Timestamp in ms
})

// Notification Queue Table
export const notificationQueue = pgTable('notification_queue', {
  id: uuid('id').defaultRandom().primaryKey(),
  type: varchar('type', { length: 50 }).notNull(), // 'discord', 'email', etc.
  payload: text('payload').notNull(), // JSON string
  attempts: integer('attempts').default(0).notNull(),
  status: varchar('status', { length: 20 }).default('pending').notNull(), // 'pending', 'processing', 'completed', 'failed'
  lastError: text('last_error'),
  nextRetry: timestamp('next_retry').defaultNow().notNull(),
  createdAt: timestamp('created_at').defaultNow().notNull(),
  updatedAt: timestamp('updated_at').defaultNow().notNull(),
}, (table) => ({
  statusIdx: index('notification_queue_status_idx').on(table.status),
  nextRetryIdx: index('notification_queue_next_retry_idx').on(table.nextRetry),
}))

export type RateLimit = typeof rateLimits.$inferSelect
export type NotificationQueueItem = typeof notificationQueue.$inferSelect
export type NewNotificationQueueItem = typeof notificationQueue.$inferInsert
